<?php
namespace boru\dhsession\handlers;

use boru\dhsession\Session;

class AdodbHandler extends HandlerInterface {

    private $db;

    private $gcAfterSessions = 10000;
    private $gcAfterMinutes = 60;
    
    public function __construct($options=[]) {
        if(isset($options["db"])) {
            $this->db = $options["db"];
            unset($options["db"]);
        } else {
            throw new \Exception("db not set");
        }
        if(isset($options["gcAfterSessions"])) {
            $this->gcAfterSessions = $options["gcAfterSessions"];
            unset($options["gcAfterSessions"]);
        }
        if(isset($options["gcAfterMinutes"])) {
            $this->gcAfterMinutes = $options["gcAfterMinutes"];
            unset($options["gcAfterMinutes"]);
        }
        parent::__construct($options);
    }

    public function init() {
        $sql = "CREATE TABLE IF NOT EXISTS ".$this->getTableName()." (
            `sessionid` varchar(255) NOT NULL,
            `ip` varchar(100) NOT NULL,
            `useragent` varchar(255) NOT NULL,
            `metaid` varchar(100) NOT NULL,
            `data` longtext NOT NULL,
            `modified` int NOT NULL,
            PRIMARY KEY (`sessionid`),
            KEY `idx_ip` (`ip`),
            KEY `idx_metaid` (`metaid`),
            KEY `idx_agent` (`useragent`),
            KEY `idx_modified` (`modified`)
        )";
        $this->db->query($sql);
        $sql = "CREATE TABLE IF NOT EXISTS ".$this->getTableName()."_seq (
            `id` int(3) NOT NULL DEFAULT 1,
            `sessions` bigint(19) NOT NULL DEFAULT 0,
            `last_gc` datetime NOT NULL,
            PRIMARY KEY (`id`)
        )";
        $this->db->query($sql);
        return parent::init();
    }
    /**
     * @return bool
     */
    public function refresh() {
        $currentId = session_id();
        $newId = session_regenerate_id();
        $sql = "update ".$this->getTableName()." set sessionid = ?, modified = ? where sessionid = ?";
        $this->db->pquery($sql, [$newId, time(), $currentId]);
        $this->setSessionId($newId);
        return true;
    }

    /**
     * @return bool
     */
    public function close() {
        $this->closeCallback();
        return $this->gc();
    }

    /**
     * @return bool
     */
    public function destroy($id) {
        $sql = "delete from ".$this->getTableName()." where sessionid = ?";
        $this->db->pquery($sql, [$id]);
        return true;
    }

    /**
     * @return bool
     */
    public function gc($max_lifetime = null) {
        $this->incrementGc();
        if($this->shouldGc()) {
            $sql = "delete from ".$this->getTableName()." where modified < ?";
            $this->db->pquery($sql, [$this->getExpireTime($max_lifetime)]);
            $this->resetGc();
        }
        return true;
    }

    /**
     * @return bool
     */
    public function open($path, $name) {
        $sql = "delete from ".$this->getTableName()." where sessionid = ? AND modified < ?";
        $this->db->pquery($sql, [$this->getSessionId(), $this->getExpireTime()]);
        $this->setSessionName($name);
        return true;
    }

    /**
     * @return string|false
     */
    public function read($id) {
        $sql = "select data from ".$this->getTableName()." where sessionid = ?";
        $result = $this->db->pquery($sql, [$id]);
        if ($result && $row = $this->db->fetchAssoc($result)) {
            $this->readCallback(Session::unserialize($row["data"]));
            return $row["data"];
        }
        if(empty($this->getSessionId())) {
            $this->setSessionId(session_id());
        }
        return "";
    }

    /**
     * @return bool
     */
    public function write($id, $data) {
        $row = $this->readRaw($id);
        if($row && !empty($row["data"])) {
            $prevData = Session::unserialize($row["data"]);
            $newData = Session::unserialize($data);
            $deletes = Session::getSessionDeletes();
            if(!empty($deletes)) {
                foreach($deletes as $delete) {
                    unset($prevData[$delete]);
                }
            }
            $_SESSION = array_merge($prevData, $newData);
            $data = session_encode();
        }
        $sql = "insert into ".$this->getTableName()." (sessionid, ip, `useragent`, `metaid`, data, modified) values (?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE data = ?, modified = ?, metaid=?";
        $this->db->pquery($sql, [$id, $_SERVER["REMOTE_ADDR"], $_SERVER['HTTP_USER_AGENT'], $this->getMetaId(), $data, time(), $data, time(), $this->getMetaId()]);
        return true;
    }
    /**
     * @return false|array
     */
    public function readRaw($id) {
        $sql = "select data from ".$this->getTableName()." where sessionid = ?";
        $result = $this->db->pquery($sql, [$id]);
        if ($result && $row = $this->db->fetchAssoc($result)) {
            return $row;
        }
        return false;
    }

    public function setMetaId($metaId) {
        $sql = "update ".$this->getTableName()." set metaid = ? where sessionid = ?";
        $this->db->pquery($sql, [$metaId, $this->getSessionId()]);
        parent::setMetaId($metaId);
    }

    public function listSessions() {
        $sql = "select * from ".$this->getTableName();
        $result = $this->db->query($sql);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }
    public function listByIp($ip) {
        $sql = "select * from ".$this->getTableName()." where ip = ?";
        $result = $this->db->pquery($sql, [$ip]);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }
    public function listByUserAgent($userAgent) {
        $sql = "select * from ".$this->getTableName()." where useragent = ?";
        $result = $this->db->pquery($sql, [$userAgent]);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }
    public function listByMetaId($metId) {
        $sql = "select * from ".$this->getTableName()." where metaid = ?";
        $result = $this->db->pquery($sql, [$metId]);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }

    private function incrementGc() {
        $sql = "insert into ".$this->getTableName()."_seq (`id`,`last_gc`) VALUES (1,NOW()) ON DUPLICATE KEY UPDATE `sessions` = `sessions` + 1";
        $this->db->query($sql);
    }
    private function shouldGc() {
        $sql = "select * from ".$this->getTableName()."_seq where `last_gc` < DATE_SUB(NOW(), INTERVAL ? MINUTE) OR `sessions` > ? LIMIT 1";
        $result = $this->db->pquery($sql, [$this->gcAfterMinutes, $this->gcAfterSessions]);
        if ($row = $this->db->next($result)) {
            return true;
        }
        return false;
    }
    private function resetGc() {
        $sql = "update ".$this->getTableName()."_seq set `sessions` = 0, `last_gc` = NOW()";
        $this->db->query($sql);
        $sql = "optimize table ".$this->getTableName();
        $this->db->query($sql);
        $sql = "optimize table ".$this->getTableName()."_seq";
        $this->db->query($sql);
    }
}