<?php
namespace boru\dhttp;

use boru\dhttp\core\Options;
use boru\dhutils\dhGlobal;
use boru\dhttp\core\Response;
use boru\dhttp\core\Request;
use Exception;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;
use boru\dhttp\middleware\CallbackMiddleware;
use boru\dhttp\middleware\HmacAuthentication;
use boru\dhttp\middleware\MiddlewareInterface;
use React\Http\Message\ResponseException;

class Client {
    /** @var HttpClient */
    private static $client;
    public static function init() {
        if(!self::$client) {
            self::$client = new HttpClient();
        }
    }

    /**
     * Get the global HttpClient object
     * @param HttpClient $client (optional) If provided, set the global HttpClient object to this
     * @return HttpClient 
     */
    public static function httpClient($client=null) {
        if(!is_null($client)) {
            self::$client = $client;
        } else {
            self::init();
        }
        return self::$client;
    }

    /**
     * Set the client options
     * @param array $options
     * @return HttpClient 
     */
    public static function setOptions($options=[]) {
        self::init();
        self::$client->setOptions($options);
        return self::$client;
    }

    /**
     * Set verifyPeer to true or false
     * @param bool $verify 
     * @return HttpClient 
     */
    public static function verifyPeer($verify=true) {
        self::init();
        self::$client->verifyPeer($verify);
        return self::$client;
    }

    /**
     * Add a middleware to the stack to be used on all requests
     * @param MiddlewareInterface $middleware 
     * @return HttpClient 
     */
    public static function addRequestMiddleware(MiddlewareInterface $middleware) {
        self::init();
        self::$client->addRequestMiddleware($middleware);
        return self::$client;
    }
    /**
     * Add Mmiddleware to the stack to be executed on the response
     * @param MiddlewareInterface $middleware 
     * @return HttpClient 
     */
    public static function addResponseMiddleware(MiddlewareInterface $middleware) {
        self::init();
        self::$client->addResponseMiddleware($middleware);
        return self::$client;
    }
    /**
     * Set the middleware stack to be used on all requests
     * @param array $middleware 
     * @return HttpClient 
     */
    public static function setRequestMiddleware($middleware) {
        self::init();
        self::$client->setRequestMiddleware($middleware);
        return self::$client;
    }
    /**
     * Set the middleware stack to be executed on the response
     * @param array $middleware 
     * @return HttpClient 
     */
    public static function setResponseMiddleware($middleware) {
        self::init();
        self::$client->setResponseMiddleware($middleware);
        return self::$client;
    }
    /**
     * Set the maximum number of concurrent requests
     * @param int $max 
     * @return HttpClient 
     */
    public static function maxConcurrency($max) {
        self::init();
        self::$client->maxConcurrency($max);
        return self::$client;
    }
    
    /** 
     * @param Request $request
     * @return \React\Promise\PromiseInterface
     */
    public static function send($request) {
        self::init();
        return self::$client->send($request);
    }
    /**
     * @param int $id The request ID to get the response for
     * @return Response|PromiseInterface|Exception 
     */
    public static function getResponse($id) {
        self::init();
        return self::$client->getResponse($id);
    }
    public static function awaitAll() {
        self::init();
        return self::$client->awaitAll();
    }

    /**
     * Create an Options object for use with the Request objects
     * * See the Options class for more information and all available parameters
     * * Short list:
     * * * form - array of form parameters
     * * * json - array of json parameters
     * * * headers - array of headers
     * * * query - array of query parameters
     * * * raw - string of body content
     * * * async - bool, whether to return a promise or not
     * * * throwException - bool, whether to throw an exception on error or not
     * @param array $options $options An array of guzzle-style options (async=>false, form=>[], json=>[], headers=>[], etc..)
     * @return Options
     */
    public static function options($options=[]) {
        self::init();
        return self::$client->options($options);
    }

    /** 
     * A shortcut to create a form Options object
     * @param array $data An array of form data
     * @param Options $options (optional) An existing Options object to add the form data to
     */
    public static function form($data=[],$options=null) {
        self::init();
        return self::$client->form($data,$options);
    }

    /** 
     * A shortcut to create a json Options object
     * @param array $data An array of json data
     * @param Options $options (optional) An existing Options object to add the json data to
     */
    public static function json($data=[],$options=null) {
        self::init();
        return self::$client->json($data,$options);
    }

    /**
     * Creates an authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $header 
     * @param mixed $token 
     * @return MiddlewareInterface
     */
    public static function authToken($token,$options=null) {
        self::init();
        return self::$client->authToken($token,$options);
    }

    /**
     * Creates a bearer authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $token 
     * @return MiddlewareInterface
     */
    public static function authBearer($token,$options=null) {
        self::init();
        return self::$client->authBearer($token,$options);
    }

    /**
     * Creates a basic authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $username 
     * @param mixed $password 
     * @return MiddlewareInterface
     */
    public static function authBasic($userPass=[],$options=null) {
        self::init();
        return self::$client->authBasic($userPass,$options);
    }

    /**
     * Creates an HMAC authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $apiKey 
     * @param mixed $apiSecret 
     * @param array $options 
     * @return HmacAuthentication 
     */
    public static function hmacAuthenticator($key,$options=null) {
        self::init();
        return self::$client->hmacAuthenticator($key,$options);
    }

    /**
     * Send the request and if async=true, return a promise. Otherwise await the return and return the Response
     * @param mixed $method ["GET","POST","PUT","DELETE","PATCH","HEAD"]
     * @param mixed $url The URl to request from
     * @param array $options An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function rquest($method,$url,$options=null) {
        self::init();
        return self::$client->request($method,$url,$options);
    }

    /**
     * Send a GET request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function get($url,$options=null,$async=null) {
        self::init();
        return self::$client->get($url,$options,$async);
    }

    /**
     * Send a POST request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function post($url,$options=null,$async=null) {
        self::init();
        return self::$client->post($url,$options,$async);
    }

    /**
     * Send a PUT request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function put($url,$options=null,$async=null) {
        self::init();
        return self::$client->put($url,$options,$async);
    }

    /**
     * Send a PATCH request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function patch($url,$options=null,$async=null) {
        self::init();
        return self::$client->patch($url,$options,$async);
    }

    /**
     * Send a DELETE request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function delete($url,$options=null,$async=null) {
        self::init();
        return self::$client->delete($url,$options,$async);
    }

    /**
     * Send a HEAD request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function head($url,$options=null,$async=null) {
        self::init();
        return self::$client->head($url,$options,$async);
    }
}
