<?php
namespace boru\dhttp;

use boru\dhttp\core\Options;
use boru\dhutils\dhGlobal;
use boru\dhttp\core\Response;
use boru\dhttp\core\Request;
use Exception;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;
use boru\dhttp\middleware\CallbackMiddleware;
use boru\dhttp\middleware\HmacAuthentication;
use boru\dhttp\middleware\MiddlewareInterface;
use React\Http\Message\ResponseException;

class HttpClient {
    private $verifyPeer = true;
    private $verifyPeerName = true;
    private $debugInfo = false;
    private $debugError = false;

    /** @var \React\Http\Browser */
    private $browserClient;
    /** @var \React\Socket\Connector */
    private $connector;

    private $options = ["headers"=>["User-Agent" => "boru/dhttp",]];

    /** @var int */
    private $maxConcurrency=3;
    /** @var \React\Promise\Deferred[] */
    private $promises=[];
    /** @var \boru\dhutils\http\Response[]|\React\Promise\PromiseInterface[]|Exception[] */
    private $results=[];

    public function __construct() {
        $this->connector = new \React\Socket\Connector([
            'tcp' => true,
            'tls' => [
                'verify_peer' => $this->verifyPeer,
                'verify_peer_name' => $this->verifyPeerName,
            ],
        ]);
        $this->browserClient = new \React\Http\Browser($this->connector);
        $this->browserClient = $this->browserClient->withRejectErrorResponse(false);
    }

    public function setOptions($options=[]) {
        $this->options = $options;
    }
    public function verifyPeer($verifyPeer=true,$verifyPeerName=true) {
        $this->verifyPeer = $verifyPeer;
        $this->verifyPeerName = $verifyPeerName;
    }
    /**
     * 
     * @param MiddlewareInterface $middleware 
     * @return void 
     */
    public function addRequestMiddleware($middleware) {
        if(is_array($middleware)) {
            foreach($middleware as $m) {
                $this->addRequestMiddleware($m);
            }
            return;
        }
        if(is_callable($middleware)) {
            $middleware = new CallbackMiddleware($middleware);
        }
        if(!($middleware instanceof MiddlewareInterface)) {
            throw new Exception("Invalid middleware");
        }
        $this->options["requestMiddleware"][] = $middleware;
    }
    public function addResponseMiddleware($middleware) {
        if(is_array($middleware)) {
            foreach($middleware as $m) {
                $this->addResponseMiddleware($m);
            }
            return;
        }
        if(is_callable($middleware)) {
            $middleware = new CallbackMiddleware($middleware);
        }
        if(!($middleware instanceof MiddlewareInterface)) {
            throw new Exception("Invalid middleware");
        }
        $this->options["responseMiddleware"][] = $middleware;
    }
    public function setRequestMiddleware($middleware=[]) {
        if(empty($middleware)) {
            $this->options["requestMiddleware"] = [];
            return;
        }
        $this->options["requestMiddleware"] = [];
        $this->addRequestMiddleware($middleware);
    }
    public function setResponseMiddleware($middleware=[]) {
        if(empty($middleware)) {
            $this->options["responseMiddleware"] = [];
            return;
        }
        $this->options["responseMiddleware"] = [];
        $this->addResponseMiddleware($middleware);
    }
    public function maxConcurrency($maxConcurrency) {
        $this->maxConcurrency = $maxConcurrency;
    }
    private function registerDeferred($id,$deferred) {
        $this->promises[$id] = $deferred;
    }
    private function registerPromise($id,$promise) {
        $this->results[$id] = $promise;
    }
    /** 
     * @param Request $request
     * @return \React\Promise\PromiseInterface
     */
    public function send($request) {
        $this->registerDeferred($request->getId(),$request->getDeferred());
        $this->applyMiddleware($request,$request->getRequestMiddleware());
        $promise = $this->browserClient->request($request->getMethod(),$request->getFullUrl(),$request->getHeaders(),$request->getBody());
        $this->registerPromise($request->getId(),$promise);
        $promise->then(function(\Psr\Http\Message\ResponseInterface $response) use ($request) {
            $this->processSuccess($request,$response);
        },function(Exception $e) use ($request) {
            $this->processException($request,$e);
        });
        $this->checkConcurrency();
        return $promise;
    }

    private function processSuccess($request,$response) {
        if($this->debugInfo) {
            dhGlobal::info("dhBrowser response from",$request->getMethod(),$request->getUrl());
        }
        $resp = new Response($response,$request);
        $this->processResponse($request,$resp);
        return $resp;
    }
    private function processException($request,$e) {
        if($this->debugError) {
            dhGlobal::error("dhBrowser error from",$request->getMethod(),$request->getUrl(),"-",$e->getMessage());
        }
        if($request->getThrowException() && $e instanceof \Exception) {
            $this->processResponse($request,$e);
            throw $e;
        }
        if($e instanceof ResponseException) {
            $resp = new Response($e->getResponse(),$request);
            return $resp;
        }
        $this->processResponse($request,$e);
        return $e;
    }
    /**
     * @param Request $request
     * @param mixed $response Response
     * @return Response
     * @throws ResponseException
     * @throws \Exception
     */
    private function processResponse($request,$response) {
        if(!isset($this->promises[$request->getId()])) {
            return $response;
        }
        $deferred = $this->promises[$request->getId()];
        unset($this->promises[$request->getId()]);

        //throw if we're supposed to, before middleware
        if($response instanceof \Exception) {
            $this->handleResponseException($request,$response);
        }

        if(!($response instanceof Response)) {
            if($response instanceof ResponseException || method_exists($response,"getResponse")) {
                $response = new Response($response->getResponse(),$request);
            } elseif($response instanceof \Exception) {
                $response = Response::fromException($response,$request);
            }
        }
        try {
            $this->applyMiddleware($response,$request->getResponseMiddleware());
        } catch(\Exception $e) {
            //throw if we're supposed to, after middleware
            $this->handleResponseException($request,$response);
        }

        //throw if we're supposed to, after middleware -- final check
        if($response instanceof \Exception) {
            $this->handleResponseException($request,$response);
        }

        $this->assignResponse($request->getId(),$response);

        if($response instanceof \Exception) {
            $deferred->reject($response);
        } else {
            $deferred->resolve($response);
        }
        return $response;
    }
    private function handleResponseException($request,$e) {
        if($request->getThrowException()) {
            $this->assignResponse($request->getId(),$e);
            throw $e;
        }
        return $e;
    }
    private function assignResponse($requestId,$response) {
        $this->results[$requestId] = $response;
    }
    private function applyMiddleware(&$object,$middlewareList=[]) {
        if(empty($middlewareList)) {
            return;
        }
        $queue = new \SplQueue();
        reset($middlewareList);
        foreach($middlewareList as $middleware) {
            $queue->enqueue($middleware);
        }
        $next = function($obj) use ($queue,&$next) {
            if($queue->isEmpty()) {
                return $obj;
            }
            $middleware = $queue->dequeue();
            return $middleware($obj,$next);
        };
        $object = $next($object);
    }

    /**
     * @return Response|PromiseInterface|Exception 
     */
    public function getResponse($id) {
        return isset($this->results[$id]) ? $this->results[$id] : false;
    }

    private function checkConcurrency() {
        if(!$this->isOverConcurrency()) {
            return;
        }
        Loop::addPeriodicTimer(0.1,function($timer) {
            if(!$this->isOverConcurrency()) {
                Loop::cancelTimer($timer);
                Loop::stop();
            }
        });
        Loop::run();
    }
    private function isOverConcurrency() {
        if(count($this->promises) < $this->maxConcurrency) {
            return false;
        }
        return true;
    }
    public function awaitAll() {
        if(!empty($this->promises)) {
            Loop::addPeriodicTimer(0.1,function($timer) {
                if(empty($this->promises)) {
                    Loop::cancelTimer($timer);
                    Loop::stop();
                }
            });
            Loop::run();
        }
    }

    /**
     * Create an Options object for use with the Request objects
     * * See the Options class for more information and all available parameters
     * * Short list:
     * * * form - array of form parameters
     * * * json - array of json parameters
     * * * headers - array of headers
     * * * query - array of query parameters
     * * * raw - string of body content
     * * * async - bool, whether to return a promise or not
     * * * throwException - bool, whether to throw an exception on error or not
     * @param array $options $options An array of guzzle-style options (async=>false, form=>[], json=>[], headers=>[], etc..)
     * @return Options
     */
    public function options($options=[]) {
        return new Options($options);
    }
    /** 
     * A shortcut to create a form Options object
     * @param array $data An array of form data
     * @param Options $options (optional) An existing Options object to add the form data to
     */
    public function form($data=[],$options=null) {
        if(!$options instanceof Options) {
            $options = $this->options($options);
        }
        $options->form($data);
        return $options;
    }
    /** 
     * A shortcut to create a json Options object
     * @param array $data An array of json data
     * @param array|Options $options (optional) An existing Options object (or constructor array) to add the json data to
     */
    public function json($data=[],$options=null) {
        if(!$options instanceof Options) {
            $options = $this->options($options);
        }
        $options->json($data);
        return $options;
    }

    /**
     * Creates an authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $header 
     * @param mixed $token 
     * @return MiddlewareInterface
     */
    public function authToken($header,$token) {
        return function($request,$next) use ($header,$token) {
            $request->header($header,$token);
            return $next($request);
        };
    }
    /**
     * Creates a bearer authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $token 
     * @return MiddlewareInterface
     */
    public function authBearer($token) {
        return function($request,$next) use ($token) {
            $request->header("Authorization","Bearer ".$token);
            return $next($request);
        };
    }
    /**
     * Creates a basic authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $username 
     * @param mixed $password 
     * @return MiddlewareInterface
     */
    public function authBasic($username,$password) {
        return function($request,$next) use ($username,$password) {
            dhGlobal::outLine("Adding header api-test");
            $request->header("Authorization","Basic: ".base64_encode($username.":".$password));
            return $next($request);
        };
    }
    /**
     * Creates an HMAC authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $apiKey 
     * @param mixed $apiSecret 
     * @param array $options 
     * @return HmacAuthentication 
     */
    public function hmacAuthenticator($apiKey,$apiSecret,$options=[
        "hashAlgo"=>"sha1",
        "headerApiKey"=>"x-api-key",
        "headerApiNonce"=>"x-api-nonce",
        "headerApiSign"=>"x-api-sign",
        "signatureTemplate"=>"{APIKEY}:{NONCE}:{PATH}",
        "withLeadingSlash"=>true,
    ]) {
        return new HmacAuthentication($apiKey,$apiSecret,$options);
    }
    /**
     * Send the request and if async=true, return a promise. Otherwise await the return and return the Response
     * @param mixed $method ["GET","POST","PUT","DELETE","PATCH","HEAD"]
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public function request($method,$url,$options=[],$async=null) {
        $staticOpts = new Options($this->options);
        if(!is_null($async)) {
            $options = $this->applyToOptions($options,["async"=>$async]);
        }
        $options = $staticOpts->merge($options)->toArray();
        if(is_null($async) && isset($options["async"])) {
            $async = $options["async"];
        } else {
            $async = $async ? true : false;
        }
        $options['method'] = $method;
        $options["url"] = $url;
        $options["async"] = $async;
        $request = new Request($options);
        $result = $request->send();
        if($async) {
            return $result;
        }
        $request->await();
        return $request->response();
    }
    /**
     * Send a GET request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public function get($url,$options=[],$async=null) {
        return $this->request("GET",$url,$options,$async);
    }
    /**
     * Send a POST request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public function post($url,$options=[],$async=null) {
        return $this->request("POST",$url,$options,$async);
    }
    /**
     * Send a PUT request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public function put($url,$options=[],$async=null) {
        return $this->request("PUT",$url,$options,$async);
    }
    /**
     * Send a DELETE request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public function delete($url,$options=[],$async=null) {
        return $this->request("DELETE",$url,$options,$async);
    }
    /**
     * Send a PATCH request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public function patch($url,$options=[],$async=null) {
        return $this->request("PATCH",$url,$options,$async);
    }
    /**
     * Send a HEAD request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public function head($url,$options=[],$async=null) {
        return $this->request("HEAD",$url,$options,$async);
    }

    private function applyToOptions($options,$optArray=[]) {
        if(empty($optArray)) {
            return $options;
        }
        if(is_array($options)) {
            $options = new Options($options);
        }
        $options->merge($optArray);
        return $options;
    }
}