<?php
namespace boru\dhttp;

use boru\dhttp\core\Options;
use boru\dhutils\dhGlobal;
use boru\dhttp\core\Response;
use boru\dhttp\core\Request;
use Exception;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;
use boru\dhttp\middleware\CallbackMiddleware;
use boru\dhttp\middleware\HmacAuthentication;
use boru\dhttp\middleware\MiddlewareInterface;
use React\Http\Message\ResponseException;

class OldClient {
    private static $verifyPeer = true;
    private static $verifyPeerName = true;

    private static $debugInfo = false;
    private static $debugError = false;

    //internals per instance

    /** @var \React\Http\Browser */
    private static $browserClient;
    /** @var \React\Socket\Connector */
    private static $Connector;
    
    // internals static
    private static $options = ["headers"=>["User-Agent" => "boru/dhttp",]];

    /** @var int */
    private static $maxConcurrency=3;
    /** @var \React\Promise\Deferred[] */
    private static $promises=[];
    /** @var \boru\dhutils\http\Response[]|\React\Promise\PromiseInterface[]|Exception[] */
    private static $results=[];

    public static function init($force=false) {
        if(is_null(static::$browserClient) || $force) {
            static::$Connector = new \React\Socket\Connector([
                "tls"=>[
                    'verify_peer' => static::$verifyPeer,
                    'verify_peer_name' => static::$verifyPeerName,
                ]
            ]);
            static::$browserClient = new \React\Http\Browser(static::$Connector);
            static::$browserClient = static::$browserClient->withRejectErrorResponse(false);
        }
    }
    public static function setOptions($options=[]) {
        static::$options = $options;
    }
    public static function verifyPeer($verifyPeer=true,$verifyPeerName=true) {
        static::$verifyPeer = $verifyPeer;
        static::$verifyPeerName = $verifyPeerName;
    }
    /**
     * 
     * @param MiddlewareInterface $middleware 
     * @return void 
     */
    public static function addRequestMiddleware($middleware) {
        if(is_array($middleware)) {
            foreach($middleware as $m) {
                static::addRequestMiddleware($m);
            }
            return;
        }
        if(is_callable($middleware)) {
            $middleware = new CallbackMiddleware($middleware);
        }
        if(!($middleware instanceof MiddlewareInterface)) {
            throw new Exception("Invalid middleware");
        }
        static::$options["requestMiddleware"][] = $middleware;
    }
    public static function addResponseMiddleware($middleware) {
        if(is_array($middleware)) {
            foreach($middleware as $m) {
                static::addResponseMiddleware($m);
            }
            return;
        }
        if(is_callable($middleware)) {
            $middleware = new CallbackMiddleware($middleware);
        }
        if(!($middleware instanceof MiddlewareInterface)) {
            throw new Exception("Invalid middleware");
        }
        static::$options["responseMiddleware"][] = $middleware;
    }
    public static function setRequestMiddleware($middleware=[]) {
        if(empty($middleware)) {
            static::$options["requestMiddleware"] = [];
            return;
        }
        static::$options["requestMiddleware"] = [];
        static::addRequestMiddleware($middleware);
    }
    public static function setResponseMiddleware($middleware=[]) {
        if(empty($middleware)) {
            static::$options["responseMiddleware"] = [];
            return;
        }
        static::$options["responseMiddleware"] = [];
        static::addResponseMiddleware($middleware);
    }
    public static function maxConcurrency($maxConcurrency) {
        static::$maxConcurrency = $maxConcurrency;
    }
    private static function registerDeferred($id,$deferred) {
        static::$promises[$id] = $deferred;
    }
    private static function registerPromise($id,$promise) {
        static::$results[$id] = $promise;
    }
    /** 
     * @param Request $request
     * @return \React\Promise\PromiseInterface
     */
    public static function send($request) {
        static::init();
        static::registerDeferred($request->getId(),$request->getDeferred());
        static::applyMiddleware($request,$request->getRequestMiddleware());
        $promise = static::$browserClient->request($request->getMethod(),$request->getFullUrl(),$request->getHeaders(),$request->getBody());
        static::registerPromise($request->getId(),$promise);
        $promise->then(function(\Psr\Http\Message\ResponseInterface $response) use ($request) {
            static::processSuccess($request,$response);
        },function(Exception $e) use ($request) {
            static::processException($request,$e);
        });
        static::checkConcurrency();
        return $promise;
    }

    private static function processSuccess($request,$response) {
        if(static::$debugInfo) {
            dhGlobal::info("dhBrowser response from",$request->getMethod(),$request->getUrl());
        }
        $resp = new Response($response,$request);
        static::processResponse($request,$resp);
        return $resp;
    }
    private static function processException($request,$e) {
        if(static::$debugError) {
            dhGlobal::error("dhBrowser error from",$request->getMethod(),$request->getUrl(),"-",$e->getMessage());
        }
        if($request->getThrowException() && $e instanceof \Exception) {
            static::processResponse($request,$e);
            throw $e;
        }
        if($e instanceof ResponseException) {
            $resp = new Response($e->getResponse(),$request);
            return $resp;
        }
        static::processResponse($request,$e);
        return $e;
    }
    /**
     * @param Request $request
     * @param mixed $response Response
     * @return Response
     * @throws ResponseException
     * @throws \Exception
     */
    private static function processResponse($request,$response) {
        if(!isset(static::$promises[$request->getId()])) {
            return $response;
        }
        $deferred = static::$promises[$request->getId()];
        unset(static::$promises[$request->getId()]);

        //throw if we're supposed to, before middleware
        if($response instanceof \Exception) {
            static::handleResponseException($request,$response);
        }

        if(!($response instanceof Response)) {
            if($response instanceof ResponseException || method_exists($response,"getResponse")) {
                $response = new Response($response->getResponse(),$request);
            } elseif($response instanceof \Exception) {
                $response = Response::fromException($response,$request);
            }
        }
        try {
            static::applyMiddleware($response,$request->getResponseMiddleware());
        } catch(\Exception $e) {
            //throw if we're supposed to, after middleware
            static::handleResponseException($request,$response);
        }

        //throw if we're supposed to, after middleware -- final check
        if($response instanceof \Exception) {
            static::handleResponseException($request,$response);
        }

        static::assignResponse($request->getId(),$response);

        if($response instanceof \Exception) {
            $deferred->reject($response);
        } else {
            $deferred->resolve($response);
        }
        return $response;
    }
    private static function handleResponseException($request,$e) {
        if($request->getThrowException()) {
            static::assignResponse($request->getId(),$e);
            throw $e;
        }
        return $e;
    }
    private static function assignResponse($requestId,$response) {
        static::$results[$requestId] = $response;
    }
    private static function applyMiddleware(&$object,$middlewareList=[]) {
        if(empty($middlewareList)) {
            return;
        }
        $queue = new \SplQueue();
        reset($middlewareList);
        foreach($middlewareList as $middleware) {
            $queue->enqueue($middleware);
        }
        $next = function($obj) use ($queue,&$next) {
            if($queue->isEmpty()) {
                return $obj;
            }
            $middleware = $queue->dequeue();
            return $middleware($obj,$next);
        };
        $object = $next($object);
    }

    /**
     * @return Response|PromiseInterface|Exception 
     */
    public static function getResponse($id) {
        return isset(static::$results[$id]) ? static::$results[$id] : false;
    }

    private static function checkConcurrency() {
        if(!static::isOverConcurrency()) {
            return;
        }
        Loop::addPeriodicTimer(0.1,function($timer) {
            if(!static::isOverConcurrency()) {
                Loop::cancelTimer($timer);
                Loop::stop();
            }
        });
        Loop::run();
    }
    private static function isOverConcurrency() {
        if(count(static::$promises) < static::$maxConcurrency) {
            return false;
        }
        return true;
    }
    public static function awaitAll() {
        if(!empty(static::$promises)) {
            Loop::addPeriodicTimer(0.1,function($timer) {
                if(empty(static::$promises)) {
                    Loop::cancelTimer($timer);
                    Loop::stop();
                }
            });
            Loop::run();
        }
    }

    /**
     * Create an Options object for use with the Request objects
     * * See the Options class for more information and all available parameters
     * * Short list:
     * * * form - array of form parameters
     * * * json - array of json parameters
     * * * headers - array of headers
     * * * query - array of query parameters
     * * * raw - string of body content
     * * * async - bool, whether to return a promise or not
     * * * throwException - bool, whether to throw an exception on error or not
     * @param array $options $options An array of guzzle-style options (async=>false, form=>[], json=>[], headers=>[], etc..)
     * @return Options
     */
    public static function options($options=[]) {
        return new Options($options);
    }
    /** 
     * A shortcut to create a form Options object
     * @param array $data An array of form data
     * @param Options $options (optional) An existing Options object to add the form data to
     */
    public static function form($data=[],$options=null) {
        if($options === null) {
            $options = static::options();
        }
        $options->form($data);
        return $options;
    }
    /** 
     * A shortcut to create a json Options object
     * @param array $data An array of json data
     * @param Options $options (optional) An existing Options object to add the json data to
     */
    public static function json($data=[],$options=null) {
        if($options === null) {
            $options = static::options();
        }
        $options->json($data);
        return $options;
    }

    /**
     * Creates an authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $header 
     * @param mixed $token 
     * @return MiddlewareInterface
     */
    public static function authToken($header,$token) {
        return function($request,$next) use ($header,$token) {
            $request->header($header,$token);
            return $next($request);
        };
    }
    /**
     * Creates a bearer authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $token 
     * @return MiddlewareInterface
     */
    public static function authBearer($token) {
        return function($request,$next) use ($token) {
            $request->header("Authorization","Bearer ".$token);
            return $next($request);
        };
    }
    /**
     * Creates a basic authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $username 
     * @param mixed $password 
     * @return MiddlewareInterface
     */
    public static function authBasic($username,$password) {
        return function($request,$next) use ($username,$password) {
            dhGlobal::outLine("Adding header api-test");
            $request->header("Authorization","Basic: ".base64_encode($username.":".$password));
            return $next($request);
        };
    }
    /**
     * Creates an HMAC authenticator middleware for use with addRequestMiddleware on a Request object or on the global Client object
     * @param mixed $apiKey 
     * @param mixed $apiSecret 
     * @param array $options 
     * @return HmacAuthentication 
     */
    public static function hmacAuthenticator($apiKey,$apiSecret,$options=[
        "hashAlgo"=>"sha1",
        "headerApiKey"=>"x-api-key",
        "headerApiNonce"=>"x-api-nonce",
        "headerApiSign"=>"x-api-sign",
        "signatureTemplate"=>"{APIKEY}:{NONCE}:{PATH}",
        "withLeadingSlash"=>true,
    ]) {
        return new HmacAuthentication($apiKey,$apiSecret,$options);
    }
    /**
     * Send the request and if async=true, return a promise. Otherwise await the return and return the Response
     * @param mixed $method ["GET","POST","PUT","DELETE","PATCH","HEAD"]
     * @param mixed $url The URl to request from
     * @param array $options An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function request($method,$url,$options=[],$async=null) {
        $staticOpts = new Options(static::$options);
        $options = $staticOpts->merge($options)->toArray();
        if(is_null($async) && isset($options["async"])) {
            $async = $options["async"];
        } else {
            $async = false;
        }
        $options['method'] = $method;
        $options["url"] = $url;
        $options["async"] = $async;
        $request = new Request($options);
        $result = $request->send();
        if($async) {
            return $result;
        }
        $request->await();
        return $request->response();
    }
    /**
     * Send a GET request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function get($url,$options=[],$async=null) {
        return static::request("GET",$url,$options,$async);
    }
    /**
     * Send a POST request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function post($url,$options=[],$async=null) {
        return static::request("POST",$url,$options,$async);
    }
    /**
     * Send a PUT request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function put($url,$options=[],$async=null) {
        return static::request("PUT",$url,$options,$async);
    }
    /**
     * Send a DELETE request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function delete($url,$options=[],$async=null) {
        return static::request("DELETE",$url,$options,$async);
    }
    /**
     * Send a PATCH request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function patch($url,$options=[],$async=null) {
        return static::request("PATCH",$url,$options,$async);
    }
    /**
     * Send a HEAD request
     * @param mixed $url The URl to request from
     * @param Options|array $options an Options object or An array of guzzle-style options (form=>, json=>, headers=>, etc..)
     * @param bool $async (default: true) if true, return a promise, otherwise await and return the Response object.
     * @return PromiseInterface|Response|Exception|static 
     */
    public static function head($url,$options=[],$async=null) {
        return static::request("HEAD",$url,$options,$async);
    }
}