<?php
namespace boru\dhttp\core;

use boru\dhttp\middleware\CallbackMiddleware;
use boru\dhttp\middleware\MiddlewareInterface;
use boru\dhutils\dhGlobal;
use Exception;

class Options {
    private $async = false;
    private $headers = [];
    private $method = "GET";
    private $baseUrl = "";
    private $url = "";
    private $body;
    private $bodyType = "raw"; // raw, json, form
    private $query = [];
    private $throwExceptions = true;

    private $requestMiddleware = [];
    private $responseMiddleware = [];

    public function __construct($options=[]) {
        if(!empty($options)) {
            $this->setOptions($options);
        }
    }

    public function toArray() {
        $return = [];
        if(!empty($this->async)) {
            $return["async"] = $this->async;
        }
        if(!empty($this->headers)) {
            $return["headers"] = $this->headers;
        }
        if(!empty($this->method)) {
            $return["method"] = $this->method;
        }
        if(!empty($this->baseUrl)) {
            $return["baseUrl"] = $this->baseUrl;
        }
        if(!empty($this->url)) {
            $return["url"] = $this->url;
        }
        if(!empty($this->body)) {
            $return["body"] = $this->body;
        }
        if(!empty($this->bodyType)) {
            $return["bodyType"] = $this->bodyType;
        }
        if(!empty($this->query)) {
            $return["query"] = $this->query;
        }
        if(!empty($this->throwExceptions)) {
            $return["throwExceptions"] = $this->throwExceptions;
        }
        if(!empty($this->requestMiddleware)) {
            $return["requestMiddleware"] = $this->requestMiddleware;
        }
        if(!empty($this->responseMiddleware)) {
            $return["responseMiddleware"] = $this->responseMiddleware;
        }
        return $return;
    }

    /**
     * Set options from an array or Options object
     * @param array|Options $options 
     * @return $this 
     */
    public function setOptions($options=[]) {
        if($options instanceof Options) {
            $options = $options->toArray();
        }
        $this->optionsToValues($options);
        return $this;
    }
    public function options($options=[]) {
        return $this->setOptions($options);
    }

    public function merge($options=[]) {
        if($options instanceof Options) {
            $options = $options->toArray();
        }
        $this->optionsToValues($options);
        return $this;
    }

    public function setMethod($method) {
        $this->method = strtoupper($method);
        return $this;
    }
    public function method($method) {
        return $this->setMethod($method);
    }

    public function setUrl($url) {
        $this->url = $url;
        return $this;
    }
    public function url($url) {
        return $this->setUrl($url);
    }
    public function setBaseUrl($url) {
        $this->baseUrl = $url;
        return $this;
    }
    public function baseUrl($url) {
        return $this->setBaseUrl($url);
    }

    public function query($keyPath,$value=null,$separator=".") {
        if(is_array($keyPath)) {
            $this->query = $keyPath;
            return $this;
        }
        dhGlobal::dotAssign($this->query,$keyPath,$value,$separator);
        return $this;
    }
    public function setHeaders($headers=[]) {
        $this->headers = $headers;
        return $this;
    }
    /**
     * Set a header or many headers
     * @param array|string $key 
     * @param mixed $value 
     * @return $this 
     */
    public function headers($key,$value=null) {
        return $this->header($key,$value);
    }
    /**
     * Set a header or many headers
     * @param array|string $key 
     * @param mixed $value 
     * @return $this 
     */
    public function header($key,$value=null) {
        if(is_array($key)) {
            foreach($key as $k=>$v) {
                $this->header($k,$v);
            }
        }
        $this->headers[$key]=$value;
        return $this;
    }
    /**
     * @param mixed $type One of "form","json","raw"
     * @return $this 
     */
    public function bodyType($type) {
        $this->bodyType = $type;
        return $this;
    }
    /**
     * @param mixed $type One of "form","json","raw"
     * @return $this 
     */
    public function bodyAs($type) {
        return $this->bodyType($type);
    }

    /**
     * Set the body of the request. If $keyPath is an array (or json), it will be used as the body.
     * @param array|string 
     * @return $this 
     */
    public function body($keyPath,$value=null,$separator=".") {
        if(is_null($value)) {
            if(!is_array($keyPath)) {
                $check = json_decode($keyPath,true);
                if(is_array($check)) {
                    $keyPath = $check;
                }
            }
            if(is_array($keyPath)) {
                foreach($keyPath as $k=>$v) {
                    $this->body($k,$v,$separator);
                }
            } else {
                $this->body = $keyPath;
            }
        } else {
            dhGlobal::dotAssign($this->body,$keyPath,$value,$separator);
        }
        return $this;
    }
    /**
     * Set the body of the request. If $keyPath is an array (or json), it will be used as the body.
     * * Tries to auto-set the Content-Type header based on the input type.
     * * If $key is a json string, it will be parsed and used as the body, and the Content-Type header will be set to application/json
     * * If $key is an array, it will be used as the body, and the Content-Type header will be set to application/x-www-form-urlencoded
     * * If $key is a string, it will be used as the body, and the Content-Type header will not be modified
     * @param array|string 
     * @return $this 
     */
    public function setBody($key=null,$value=null,$separator=".") {
        if(is_array($key)) {
            $this->form($key);
            return $this;
        } elseif(is_string($key)) {
            $check = json_decode($key,true);
            if(is_array($check)) {
                $this->json($check);
                return $this;
            } else {
                $this->body = $key;
                return $this;
            }
        }
        return $this->body($key,$value,$separator);
    }
    /**
     * Set the body of the request as json and set the Content-Type header to application/json
     * @param array|string 
     * @return $this 
     */
    public function json($key=null,$value=null,$separator=".") {
        $this->header('Content-Type','application/json');
        $this->bodyType = "json";
        $this->body($key,$value,$separator);
        return $this;
    }
    /**
     * Set the body of the request as form data and set the Content-Type header to application/x-www-form-urlencoded
     * @param array|string 
     * @return $this 
     */
    public function form($key=null,$value=null,$separator=".") {
        $this->header('Content-Type','application/x-www-form-urlencoded');
        $this->bodyType = "form";
        $this->body($key,$value,$separator);
        return $this;
    }
    public function raw($data) {
        $this->bodyType = "raw";
        $this->body = $data;
        return $this;
    }

    /**
     * Whether to send the request asynchronously. If true, a promise will be returned instead of the response.
     * @param bool $async (default is false)
     * @return $this 
     */
    public function async($async=true) {
        $this->async = $async;
        return $this;
    }
    /**
     * Whether to throw exceptions on error. If false, the response will be returned even if it is an error, and Exceptions will be converted to Response objects.
     * @param bool $throw (default is true);
     * @return $this 
     */
    public function throwExceptions($throw=true) {
        $this->throwExceptions = $throw;
        return $this;
    }

    
    public function authBasic($userPass=[]) {
        $this->headers["Authorization"] = "Basic ".base64_encode(implode(":",$userPass));
        return $this;
    }
    public function authToken($token) {
        $this->headers["Authorization"] = $token;
        return $this;
    }

    /**
     * Add a middleware to the Request object. Middleware will be called in the order they are added and will be passed the request object as the first parameter and the next middleware as the second parameter.
     * @param MiddlewareInterface|Callable $middleware 
     * @return $this 
     */
    public function addRequestMiddleware($middleware) {
        if(is_array($middleware)) {
            foreach($middleware as $m) {
                $this->addRequestMiddleware($m);
            }
            return;
        }
        if(is_callable($middleware)) {
            $middleware = new CallbackMiddleware($middleware);
        }
        if(!($middleware instanceof MiddlewareInterface)) {
            throw new Exception("Invalid middleware");
        }
        $this->requestMiddleware[] = $middleware;
    }
    /**
     * Add a middleware to the response. Middleware will be called in the order they are added and will be passed the response or exception object as the first parameter and the next middleware as the second parameter.
     * @param MiddlewareInterface|Callable $middleware 
     * @return $this 
     */
    public function addResponseMiddleware($middleware) {
        if(is_array($middleware)) {
            foreach($middleware as $m) {
                $this->addResponseMiddleware($m);
            }
            return;
        }
        if(is_callable($middleware)) {
            $middleware = new CallbackMiddleware($middleware);
        }
        if(!($middleware instanceof MiddlewareInterface)) {
            throw new Exception("Invalid middleware");
        }
        $this->responseMiddleware[] = $middleware;
    }
    public function setRequestMiddleware($middleware=[]) {
        if(empty($middleware)) {
            $this->requestMiddleware = [];
            return;
        }
        $this->requestMiddleware = [];
        $this->addRequestMiddleware($middleware);
    }
    public function setResponseMiddleware($middleware=[]) {
        if(empty($middleware)) {
            $this->responseMiddleware = [];
            return;
        }
        $this->responseMiddleware = [];
        $this->addResponseMiddleware($middleware);
    }


    private function optionsToValues($options=[]) {
        if(!is_array($options) || empty($options)) {
            return;
        }
        foreach($options as $think=>$value) {
            switch($think) {
                case "body":
                    $this->body($value);
                    break;
                case "json":
                    $this->json($value);
                    break;
                case "form":
                    $this->form($value);
                    break;
                case "raw":
                    $this->raw($value);
                    break;
                case "query":
                    $this->query($value);
                    break;
                case "headers":
                    foreach($value as $k=>$v) {
                        $this->header($k,$v);
                    }
                    break;
                case "auth":
                    if(is_array($value)) {
                        $this->authBasic($value);
                    } else {
                        $this->authToken($value);
                    }
                    break;
                case "bodyType":
                    $this->bodyType($value);
                    break;
                case "as":
                    $this->bodyType($value);
                    break;
                case "method":
                    $this->setMethod($value);
                    break;
                case "url":
                    $this->setUrl($value);
                    break;
                case "baseUrl":
                    $this->setBaseUrl($value);
                    break;
                case "async":
                    $this->async($value);
                    break;
                case "requestMiddleware":
                    $this->setRequestMiddleware($value);
                    break;
                case "responseMiddleware":
                    $this->setResponseMiddleware($value);
                    break;
                case "throwExceptions":
                    $this->throwExceptions($value);
                    break;
            }
        }
    }

    public function getMethod() {
        return $this->method;
    }
    public function getUrl() {
        return $this->url;
    }
    public function getUrlParts() {
        return parse_url($this->getFullUrl());
    }
    public function getFullUrl() {
        $url = $this->url;
        if(strtolower(substr($url,0,4)) != "http") {
            $preurl = $this->baseUrl;
            if(substr($preurl,-1) != "/" && substr($url,0,1) != "/") {
                $preurl .= "/";
            }
            $url = $preurl.$url;
        }
        if(!empty($this->query)) {
            if(strpos($url,"?")===false) {
                $url .= "?";
            } else {
                $url .= "&";
            }
            $url .= http_build_query($this->query);
        }
        return $url;
    }
    public function getHeaders() {
        return !is_array($this->headers) ? [] : $this->headers;
    }
    public function getRawBody() {
        return $this->body;
    }
    public function getBody() {
        $body = $this->body;
        if(is_array($body)) {
            if($this->bodyType == "form") {
                $body = http_build_query($body);
            } else if($this->bodyType == "json") {
                $body = json_encode($body);
            }
        }
        return $body;
    }
    public function getBodyType() {
        return $this->bodyType;
    }
    public function getQuery() {
        return $this->query;
    }
    public function getQueryAsString() {
        return http_build_query($this->query);
    }
    public function getResponseMiddleware() {
        return $this->responseMiddleware;
    }
    public function getRequestMiddleware() {
        return $this->requestMiddleware;
    }
    public function getAsync() {
        return $this->async;
    }
    public function isAsync() {
        return $this->async;
    }
    public function getThrowExceptions() {
        return $this->throwExceptions;
    }
}