<?php
namespace boru\dhttp\core;

use boru\dhttp\core\Options;
use boru\dhttp\Client;
use boru\dhttp\middleware\CallbackMiddleware;
use boru\dhttp\middleware\MiddlewareInterface;
use boru\dhutils\dhGlobal;
use Exception;
use Fig\Http\Message\RequestMethodInterface;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;

class Request implements RequestMethodInterface {
    private $id;

    /** @var Options */
    private $options;

    /** @var \React\Promise\Deferred */
    private $deferred;
    /** @var \React\Promise\PromiseInterface */
    private $promise;
    /** @var \React\HttpClient\Response */
    private $response;

    /** @var \boru\dhttp\HttpClient */
    private $httpClient = null;

    private $loopStopped = false;

    public function __construct($options=[],$client=null) {
        $this->id = uniqid();
        if($options instanceof Options) {
            $this->options = $options;
        } else if(is_array($options)) {
            $this->options = new Options($options);
        } else {
            throw new Exception("Invalid options type");
        }
        if(!is_null($client)) {
            $this->httpClient = $client;
        } else {
            $this->httpClient = Client::httpClient();
        }
        $this->deferred = new Deferred();
    }
    public function __destruct() {
        if($this->isAsync() && !$this->isDone()) {
            $this->await();
        }
        if($this->loopStopped) {
            Loop::run();
        }
    }
    public function __clone() {
        $this->id = uniqid();
        $this->deferred = new Deferred();
        $this->promise = null;
        $this->response = null;
        $this->options = clone($this->options);
    }

    public function httpClient($client=null) {
        if(!is_null($client)) {
            $this->httpClient = $client;
        }
        $this->httpClient = $client;
        return $this;
    }
    public function setOptions($options=[]) {
        $this->options->setOptions($options);
        return $this;
    }

    public function setMethod($method) {
        $this->options->setMethod($method);
        return $this;
    }

    public function setUrl($url) {
        $this->options->setUrl($url);
        return $this;
    }
    public function setBaseUrl($url) {
        $this->options->setBaseUrl($url);
        return $this;
    }

    public function query($keyPath,$value=null,$separator=".") {
        $this->options->query($keyPath,$value,$separator);
        return $this;
    }
    public function method($method) {
        $this->options->setMethod($method);
        return $this;
    }
    public function url($url) {
        $this->options->setUrl($url);
        return $this;
    }
    public function header($key,$value=null) {
        $this->options->header($key,$value);
        return $this;
    }
    /**
     * @param mixed $type One of "form","json","raw"
     * @return $this 
     */
    public function bodyType($type) {
        $this->options->bodyType($type);
        return $this;
    }
    public function bodyAs($type) {
        return $this->bodyType($type);
    }

    public function body($keyPath,$value=null,$separator=".") {
        $this->options->body($keyPath,$value,$separator);
        return $this;
    }
    public function setBody($key=null,$value=null,$separator=".") {
        $this->options->setBody($key,$value,$separator);
        return $this;
    }
    public function json($key=null,$value=null,$separator=".") {
        $this->options->json($key,$value,$separator);
        return $this;
    }
    public function form($key=null,$value=null,$separator=".") {
        $this->options->form($key,$value,$separator);
        return $this;
    }
    public function raw($data) {
        $this->options->raw($data);
        return $this;
    }

    public function async($async=true) {
        $this->options->async($async);
        return $this;
    }

    
    public function authBasic($userPass=[]) {
        $this->options->authBasic($userPass);
        return $this;
    }
    public function authToken($token) {
        $this->options->authToken($token);
        return $this;
    }

    /**
     * 
     * @param MiddlewareInterface $middleware 
     * @return void 
     */
    public function addRequestMiddleware($middleware) {
        $this->options->addRequestMiddleware($middleware);
    }
    public function addResponseMiddleware($middleware) {
        $this->options->addResponseMiddleware($middleware);
    }
    public function setRequestMiddleware($middleware=[]) {
        $this->options->setRequestMiddleWare($middleware);
    }
    public function setResponseMiddleware($middleware=[]) {
        $this->options->setResponseMiddleWare($middleware);
    }

    public function getMethod() {
        return $this->options->getMethod();
    }
    public function getUrl() {
        return $this->options->getUrl();
    }
    public function getFullUrl() {
        return $this->options->getFullUrl();
    }
    public function getHeaders() {
        return $this->options->getHeaders();
    }
    public function getRawBody() {
        return $this->options->getRawBody();
    }
    public function getBody() {
        $body = $this->options->getBody();
        if(empty($body)) {
            return '';
        }
        return $body;
    }
    public function getBodyType() {
        return $this->options->getBodyType();
    }
    public function getQuery() {
        return $this->options->getQuery();
    }
    public function getQueryAsString() {
        return http_build_query($this->getQuery());
    }
    public function getResponseMiddleware() {
        return $this->options->getResponseMiddleware();
    }
    public function getRequestMiddleware() {
        return $this->options->getRequestMiddleware();
    }
    public function getDeferred() {
        return $this->deferred;
    }
    public function getPromise() {
        return $this->promise;
    }
    public function isAsync() {
        return $this->options->isAsync();
    }
    public function getThrowExceptions() {
        return $this->options->getThrowExceptions();
    }
    public function getId() {
        return $this->id;
    }

    /**
     * Send the request
     * @param   array $options
     * @return  \React\Promise\PromiseInterface|Response|\Exception|false
     */
    public function send($options=[]) {
        $this->options->setOptions($options);
        $this->promise = $this->httpClient->send($this);
        if($this->isAsync()) {
            return $this->deferred->promise();
        }
        $this->await();
        return $this->response();
    }
    public function isDone() {
        return $this->response() !== false;
    }
    /**
     * Get the response
     * @return  Response|\Exception|false
     */
    public function response() {
        if(is_null($this->response)) {
            $response = $this->httpClient->getResponse($this->id);
            if($response instanceof PromiseInterface) {
                return false;
            }
            $this->response = $response;
        }
        return $this->response;
    }

    /**
     * Await the response and return it
     * @return  bool
     */
    public function await() {
        if($this->isDone()) {
            return $this->response();
        }
        Loop::addPeriodicTimer(0.1,function($timer) {
            if($this->isDone()) {
                Loop::cancelTimer($timer);
                Loop::stop();
                $this->loopStopped = true;
            }
        });
        Loop::run();
        return true;
    }
}