<?php
namespace boru\dhutils\async;

use \boru\dhutils\async\worker\WorkerProcess;
use \boru\dhutils\async\Work;
use \boru\dhutils\dhGlobal;
use boru\dhutils\dhOut;
use boru\dhutils\progress\QueueStatus;
use boru\dhutils\tools\StdOut;
use UnexpectedValueException;
use RuntimeException;
use Exception;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;

class Queue {
    /** @var Work[] */
    private $work=[];

    /** @var WorkGroup[] */
    private $workGroups=[];

    /** @var Work[] */
    private $queued=[],$active=[],$done=[];
    private $expected;

    private $retryCounts=[];

    /** @var \React\Promise\Deferred */
    private $deferred;
    private $isDone = false;
    private $isWaiting = false;
    private $maxIdleTime = 300;
    private $lastActivity=null;
    private $maxQueued=false;

    /** @var QueueStatus */
    private $progress;

    /** @var WorkerManager */
    private $workerManager;

    private $max=3;
    private $retriesOnError=0;
    
    private $visualize=false;
    private $workerErrorLimit=0;
    
    private $log = false;
    /** @var dhOut */
    private $logger;
    

    public function __construct($max=null,$workerBootstrap=null,$retriesOnError=null,$visualize=null,$workerErrorLimit=0,$log=null) {
        $this->max              = !is_null($max)              ? $max              : dhGlobal::get("asyncQueue.maxPerQueue",3);
        $workerBootstrap        = !is_null($workerBootstrap)  ? $workerBootstrap  : dhGlobal::get("asyncQueue.defaultBootstrap",null);
        $this->retriesOnError   = !is_null($retriesOnError)   ? $retriesOnError   : dhGlobal::get("asyncQueue.retriesOnError",0);
        $this->visualize        = !is_null($visualize)        ? $visualize        : dhGlobal::get("asyncQueue.visualize",false);
        $this->log              = !is_null($log)              ? $log              : dhGlobal::get("asyncQueue.log",false);
        $this->deferred = new Deferred();
        if($this->visualize) {
            $this->progress = new QueueStatus("Progress");
            $this->promise()->then(function() {
                $this->updateProgress();
                $this->progress->stop();
            });
            StdOut::progressBar($this->progress);
        }
        $this->workerManager = new WorkerManager($this,$this->max);
        $this->workerManager->setWorkerBootstrap($workerBootstrap);
        $this->workerManager->setLog($this->log);
    }
    public function __destruct() {
        if(!is_null($this->progress)) {
            $this->progress->stop();
        }
    }
    public function updateInterval($interval=0.5) {
        if($this->visualize && !is_null($this->progress)) {
            $this->progress->updateInterval($interval);
        }
        return $this;
    }
    public function enableExtendedBar($inline=false) {
        $this->progress->showExec(true);
        $this->progress->setThreads($this->max);
        return $this;
    }

    /**
     * Add work to the Queue
     * @param Work $work 
     * @return Work 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     * @throws Exception 
     */
    public function queue(Work $work,$groupIdentifier=null) {
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
        }
        $this->work[$work->getId()] = $work;
        if(!is_null($groupIdentifier)) {
            $this->addToWorkGroup($work,$groupIdentifier);
        }
        $this->moveToQueued($work);
        $this->workerManager()->checkWorkers(true);
        $this->checkMaxQueued();
        return $work;
    }

    /**
     * Creates a new workGroup and returns a promise. Once all work within the workGroup is completed, the promise will resolve with the groupIdentifier
     * 
     * The workGroup Identifier can be used in Queue->getWorkGroup($groupIdentifier)
     * @param mixed $groupIdentifier 
     * @return WorkGroup 
     */
    public function createWorkGroup($groupIdentifier) {
        $this->workGroups[$groupIdentifier] = new WorkGroup($groupIdentifier);
        return $this->workGroups[$groupIdentifier]->promise();
    }
    public function workGroupPromise($groupIdentifier) {
        return $this->workGroups[$groupIdentifier]->promise();
    }
    private function addToWorkGroup(Work &$work,$groupIdentifier) {
        if(!isset($this->workGroups[$groupIdentifier])) {
            $this->createWorkGroup($groupIdentifier);
        }
        $this->workGroups[$groupIdentifier]->addWork($work);
    }
    public function reset() {
        $this->deferred=null;
        $this->queued=[];
        $this->active=[];
        $this->done=[];
        $this->work=[];
        return $this;
    }
    
    /**
     * Get the next queued item, or false if nothing is queued 
     * @return Work|false */
    public function next() {
        if(!empty($this->queued)) {
            $work = array_shift($this->queued);
            $this->moveToActive($work);
            return $work;
        }
        return false;
    }

    public function promise() {
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
            $this->deferred->resolve([]);
        }
        return $this->deferred->promise();
    }
    public function wait() {
        $this->log("wait called");
        $this->isWaiting=true;
        $this->isDone=false;
        $this->promise()->then(function() use (&$isDone) {
            $this->isDone=true;
        });
        $progressTimer = Loop::addPeriodicTimer(0.5,function($t) {
            $this->updateProgress();
        });
        Loop::addPeriodicTimer(0.1,function($timer) use ($progressTimer)  {
            if($this->isDone) {
                Loop::cancelTimer($timer);
                if(!is_null($progressTimer)) {
                    Loop::cancelTimer($progressTimer);
                }
                Loop::stop();
            } else {
                $this->workerManager()->checkWorkers();
            }
        });
        while(!$this->isDone) {
            Loop::run();
        }
        //\React\Async\await($this->promise());
        return $this;
    }
    public function collect($groupIdentifier=null) {
        $this->wait();
        if(!is_null($groupIdentifier)) {
            return isset($this->workGroups[$groupIdentifier]) ? $this->workGroups[$groupIdentifier] : false;
        }
        return $this->work;
    }
    public function getWorkGroup($groupIdentifier=null) {
        return isset($this->workGroups[$groupIdentifier]) ? $this->workGroups[$groupIdentifier] : false;
    }
    private function checkMaxQueued() {
        if($this->maxQueued !== false && $this->maxQueued>0 && count($this->queued) >= $this->maxQueued) {
            Loop::addPeriodicTimer(0.1,function($timer) {
                if($this->maxQueued !== false && $this->maxQueued>0 && count($this->queued) >= $this->maxQueued) {
                } else {
                    Loop::stop();
                }
            });
            Loop::run();
        }
    }

    public function workerManager() {
        return $this->workerManager;
    }
    public function setWorkerLogDir($logDir) {
        $this->workerManager->setLogDir($logDir);
    }

    /** @return bool true if queued array is not empty, false if empty */
    public function hasQueued(){
        return !empty($this->queued);
    }
    /** @return bool true if the active array is not empty, false if empty */
    public function hasActive(){
        return !empty($this->active);
    }
    /** @return bool true if the done array is not empty, false if empty */
    public function hasDone(){
        return !empty($this->done);
    }

    /**
     * Retrieve the work record by ID
     * @param string $workId 
     * @return Work|false 
     */
    public function getWork($workId) {
        return isset($this->work[$workId]) ? $this->work[$workId] : false;
    }
    /**
     * Retrieve full work array
     * @return Work[]|false 
     */
    public function getAllWork() {
        return !empty($this->work) ? $this->work : false;
    }

    /** @return bool true if $work is in the active state, false otherwise */
    public function workIsActive(Work $work) {
        return $this->getActive($work->getId()) !== false ? true : false;
    }
    /** @return bool true if $work is in the queued state, false otherwise */
    public function workIsQueued(Work $work) {
        return $this->getActive($work->getId()) !== false ? true : false;
    }
    /** @return bool true if $work is in the done state, false otherwise */
    public function workIsDone(Work $work) {
        return $this->getDone($work->getId()) !== false ? true : false;
    }

    /**
     * Get the ueue, or get a Work from the queue by ID
     * @param string|null $workId
     * @return Work[]|false|Work
     */
    public function getQueued($workId=null) {
        if(!is_null($workId)) {
            return dhGlobal::getVal($this->queued,$workId,false);
        }
        return $this->queued;
    }

    /**
     * Get the active queue, or get a Work from the queue by ID
     * @param string|null $workId
     * @return Work[]|false|Work
     */
    public function getActive($workId=null) {
        if(!is_null($workId)) {
            return dhGlobal::getVal($this->active,$workId,false);
        }
        return $this->active;
    }

    /**
     * Get the done queue, or get a Work from the queue by ID
     * @param string|null $workId
     * @return Work[]|false|Work
     */
    public function getDone($workId=null) {
        if(!is_null($workId)) {
            return dhGlobal::getVal($this->done,$workId,false);
        }
        return $this->active;
    }

    /**
     * Set the value of visualize
     * @param   mixed  $visualize  
     * @return  self
	 */
    public function setVisualize($visualize) {
        $this->visualize = $visualize;
        return $this;
    }

    
    /**
	 * Get the value of workerErrorLimit
     * @return  mixed
     */
    public function getWorkerErrorLimit() {
        return $this->workerErrorLimit;
    }

    /**
     * Set the value of workerErrorLimit
     * @param   mixed  $workerErrorLimit  
     * @return  self
	 */
    public function setWorkerErrorLimit($workerErrorLimit) {
        $this->workerErrorLimit = $workerErrorLimit;
        $this->workerManager->setWorkerErrorLimit($this->workerErrorLimit);
        return $this;
    }

    /**
	 * Get the value of retriesOnError
     * @return  mixed
     */
    public function getRetriesOnError() {
        return $this->retriesOnError;
    }

    /**
     * Set the value of retriesOnError
     * @param   mixed  $retriesOnError  
     * @return  self
	 */
    public function setRetriesOnError($retriesOnError) {
        $this->retriesOnError = $retriesOnError;
        return $this;
    }

    /**
	 * Get the value of maxQueued
     * @return  mixed
     */
    public function getMaxQueued() {
        return $this->maxQueued;
    }

    /**
     * Set the value of maxQueued
     * @param   mixed  $maxQueued  
     * @return  self
	 */
    public function setMaxQueued($maxQueued) {
        $this->maxQueued = $maxQueued;
        return $this;
    }

    /**
	 * Get the value of expected
     * @return  int
     */
    public function getExpected() {
        return $this->expected;
    }

    /**
     * Set the value of expected
     * @param   int  $expected  
     * @return  self
	 */
    public function setExpected($expected) {
        $this->expected = $expected;
        return $this;
    }

    /**
	 * Get the value of maxWorkerWork
     * @return  int
     */
    public function getMaxWorkPerWorker() {
        return $this->workerManager->getMaxWorkPerWorker();
    }
    /**
     * Set the value of maxWorkerWork
     * @param   int $maxWorkerWork
     * @return  self
     */ 
    public function setMaxWorkPerWorker($maxWorkerWork)
    {
        $this->workerManager->setMaxWorkPerWorker($maxWorkerWork);
        return $this;
    }

    
    /**
	 * Get the value of killOnError
     * @return  mixed
     */
    public function getKillOnError() {
        return $this->workerManager->getKillOnError();
    }

    /**
     * Set the value of killOnError
     * @param   mixed  $killOnError  
     * @return  self
	 */
    public function setKillOnError($killOnError) {
        $this->workerManager->setKillOnError($killOnError);
        return $this;
    }

    /**
     * Get after this number of seconds, the worker will be closed, 0 to disable
     * @return  int
     */ 
    public function getWorkerTimeout()
    {
        return $this->workerManager->getWorkerTimeout();
    }

    /**
     * Set after this number of seconds, the worker will be closed, 0 to disable
     * @param  int  $workerTimeout  after this number of seconds, the worker will be closed, 0 to disable
     * @return  self
     */ 
    public function setWorkerTimeout($workerTimeout)
    {
        $this->workerManager->setWorkerTimeout($workerTimeout);
        return $this;
    }

    /**
     * Set the value of bootstrapAsCallable
     * @param   mixed  $bootstrapAsCallable  
     * @return  self
	 */
    public function setBootstrapAsCallable($bootstrapAsCallable) {
        $this->workerManager->setBootstrapAsCallable($bootstrapAsCallable);
        return $this;
    }
    /**
     * Get the value of bootstrapAsCallable
     * @return  bool
     */ 
    public function getBootstrapAsCallable()
    {
        return $this->workerManager->getBootstrapAsCallable();
    }

    /**
     * 
     * @param false|string $log 
     * @return self 
     */
    public function setLog($log) {
        $this->log = $log;
        $this->workerManager->setLog($log);
        return $this;
    }

    public function updateProgress() {
        if(!$this->visualize || $this->workerManager->getProcessCount()<=0) {
            return;
        }
        $this->progress->update(count($this->done),count($this->active),count($this->queued),$this->expected);
        if(!is_null($this->lastActivity) && $this->maxIdleTime>0 && microtime(true)-$this->lastActivity>=$this->maxIdleTime) {
            $time = round(microtime(true)-$this->lastActivity);
            $dtF = new \DateTime('@0');
            $dtT = new \DateTime("@$time");
            
            $str = dhGlobal::dateIntervalToElapsed($dtF->diff($dtT),true,false,2,"");
            dhGlobal::outLine("it has been",$str,"since anything happened.. closing");
            if($this->isWaiting) {
                $this->isDone=true;
            } else {
                exit();
            }
            
        }
    }
    public function updateWorkerTime($pid) {
        if(!$this->visualize || $this->workerManager->getProcessCount()<=0) {
            return;
        }
        if(($k = $this->workerManager()->getWorkerMap($pid)) !== false) {
            $this->progress->updateThread($k,"time",microtime(true));
        }
    }
    public function updateWorkerStatus($pid,$status) {
        if(!$this->visualize || $this->workerManager->getProcessCount()<=0) {
            return;
        }
        if(($k = $this->workerManager()->getWorkerMap($pid)) !== false) {
            $this->progress->updateThread($k,"display",$status);
            $this->progress->updateThread($k,"time",microtime(true));
        }
    }

    public function moveToQueued(Work $work) {
        $work->onWorkQueued();
        return $this->moveTo($work,$this->queued);
    }
    public function moveToActive(Work $work) {
        $work->onWorkStart();
        return $this->moveTo($work,$this->active);
    }
    public function moveToDone(Work $work) {
        return $this->moveTo($work,$this->done);
    }

    public function workOnDone(Work $work,$message=null) {
        $this->logWork($work,"workOnDone");
        $this->moveToDone($work);
        if(!is_null($message)) {
            $work->done($message->getDataObject());
        } else {
            $work->done(new Result([],$work->getId(),false));
        }
        
    }
    public function workOnError(Work $work,$message=null,$workerClosed=false) {
        if($workerClosed) {
            $this->logWork($work,"workOnError","worker closed...");
            if($this->retriesOnError>0) {
                dhGlobal::debug("Work stopped due to worker close, handling with retry process");
                $this->handleWorkOnError($work,$message);
            } else {
                $this->moveToQueued($work);
                dhGlobal::debug("Work re-queued due to worker close");
            }
        } else {
            $this->handleWorkOnError($work,$message);
        }
    }
    private function handleWorkOnError(Work $work,$message=null) {
        $this->logWork($work,"handleWorkOnError",!is_null($message) ? $message : "");
        $retryCount = isset($this->retryCounts[$work->getId()]) ? $this->retryCounts[$work->getId()] : 0;
        $retryCount++;
        if($this->retriesOnError>0 && $retryCount<=$this->retriesOnError) {
            dhGlobal::trace("[retry]","retry count for",$work->getId(),"is",$retryCount);
            $this->logWork($work,"RETRY #".$retryCount);
            $this->retryCounts[$work->getId()] = $retryCount;
            $this->moveToQueued($work);
        } else {
            $this->moveToDone($work);
            if(!is_null($message)) {
                $work->error($message->getErrorObject());
            } else {
                $work->error(new Result([],$work->getId(),true));
            }
        }
    }

    private function moveTo(Work $work,&$destinationArray) {
        $this->lastActivity = microtime(true);
        if(!isset($this->work[$work->getId()])) {
            throw new \Exception("Work not found in work array");
        }
        unset($this->queued[$work->getId()]);
        unset($this->active[$work->getId()]);
        unset($this->done[$work->getId()]);
        $destinationArray[$work->getId()] = &$this->work[$work->getId()];
        if(!$this->hasActive() && !$this->hasQueued()) {
            if(!is_null($this->deferred)) {
                //we need to resolve the QueuePromise after the final 'work' promise, otherwise we will resolve the QueuePromise before the WorkPromise is processed.
                $work->then(function() {
                    $this->deferred->resolve($this->done);
                },function() {
                    $this->deferred->resolve($this->done);
                });
            }
        }
        $this->updateProgress();
        return $work;
    }

    private function log(...$args) {
        array_unshift($args,"[QUEUE ]");
        $this->_log(...$args);
    }
    private function logWorker(WorkerProcess $process,...$args) {
        array_unshift($args,"[WORKER]",":".$process->getId().":");
        $this->_log(...$args);
    }
    private function logWork(Work $work,...$args) {
        array_unshift($args,"[ WORK ]",":".$work->getId().":");
        $this->_log(...$args);
    }
    private function _log(...$args) {
        if($this->log !== false) {
            if(is_null($this->logger)) {
                $this->logger = new dhOut(false,$this->log);
            }
            $this->logger->line(...$args);
        }
    }
    private function logDisplay($var,$len=150) {
        if(is_array($var)) {
            $var = json_encode($var);
        }
        if(!is_array($var) && !is_object($var)) {
            if(strlen($var)>$len) {
                $packet = substr($var,0,$len-4)." ...";
            }
        }
        return $var;
    }

    public static function callableWork($callable,...$args) {
        $work = new Work($callable);
        $work->args($args);
        $work->asJson(false);
        return $work;
    }
    public static function basicDisplay(Work $work,$identifier,...$args) {
        if(empty($args)) {
            $args = $work->args();
        }
        if(empty($args)) {
            $args = [];
        } else {
            if(!is_array($args)) {
                $args = [$args];
            }
        }
        $work->onStart(function() use ($identifier,$args) {
            Queue::display("start",$identifier,...$args);
        });
        $work->then(function(Result $result) use ($identifier) {
            Queue::display("complete",$identifier,$result->result());
        },function(Result $result) use ($identifier) {
            $resData = $result->result();
            $errData = $result->error();
            if(!empty($errData)) {
                Queue::display("error",$identifier,$errData);
            } else {
                if(empty($resData)) {
                    $resData = "-empty-";
                }
                Queue::display("error",$identifier,$resData);
            }
            
        });
        return $work;
    }
    public static function basicDisplayOnDone(Work $work,$identifier,...$args) {
        if(empty($args)) {
            $args = $work->args();
        }
        if(empty($args)) {
            $args = [];
        } else {
            if(!is_array($args)) {
                $args = [$args];
            }
        }
        $work->then(function(Result $result) use ($identifier) {
            Queue::display("complete",$identifier,$result->result());
        },function(Result $result) use ($identifier) {
            $resData = $result->result();
            $errData = $result->error();
            if(!empty($errData)) {
                Queue::display("error",$identifier,$errData);
            } else {
                if(empty($resData)) {
                    $resData = "-empty-";
                }
                Queue::display("error",$identifier,$resData);
            }
            
        });
        return $work;
    }
    public static function display($action,$identifier,...$args) {
        $displayLen = stdOut::getCols(60);
        if($displayLen<20) {
            $displayLen=20;
        }
        $prefix = $suffix = "";
        if($action == "start") {
            $prefix = "> --- started  :";
        } elseif($action == "error") {
            $prefix = "> !!! failure  :";
        } elseif($action == "complete") {
            $prefix = "> *** complete :";
        }
        $argString = "";
        if(!empty($args)) {
            foreach($args as $k=>$arg) {
                $args[$k] = StdOut::argToString($arg,true);
            }
            $argString = implode(" ",$args);
            if(strlen($argString) >= $displayLen) {
                $argString=substr($argString,0,$displayLen-4)." ...";
            }
        }
        dhGlobal::outLine($prefix,$identifier,$argString);
    }

    /**
	 * Get the value of maxIdleTime
     * @return  mixed
     */
    public function getMaxIdleTime() {
        return $this->maxIdleTime;
    }

    /**
     * Set the value of maxIdleTime
     * @param   mixed  $maxIdleTime  
     * @return  self
	 */
    public function setMaxIdleTime($maxIdleTime) {
        $this->maxIdleTime = $maxIdleTime;
        return $this;
    }
}