<?php
namespace boru\dhutils\async;

use boru\dhutils\dhGlobal;
use boru\dhutils\dhOut;
use \boru\dhutils\async\worker\WorkerProcess;

class WorkerManager {
    /** @var Queue */
    private $queue;
    /** @var WorkerProcess[] */
    private $processes=[];
    private $processWorkCounts=[];
    private $processMap=[];
    private $processWork = [];

    private $numWorkers = 2;

    private $workerBootstrap;
    private $bootstrapAsCallable=false;
    private $workerErrorLimit=0;
    private $maxWorkerWork=0;
    private $workerTimeout=0;
    private $killOnError=false;

    private $logDir;

    private $log = false;
    /** @var dhOut */
    private $logger;

    public function __construct($queue,$numWorkers=2) {
        $this->queue = $queue;
        $this->numWorkers = $numWorkers;
    }
    public function __destruct() {
        if(!empty($this->processes)) {
            foreach($this->processes as $process) {
                $process->stop();
            }
        }
    }

    public function checkWorkers($launchIfNeeded=true) {
        foreach($this->processes as $pid=>$process) {
            if(!$process->isRunning() || $process->isDone()) {
                $this->removeWorker($pid);
            } else { 
                $this->checkWorkerTimeout($process);
            }
        }
        if($launchIfNeeded) {
            $tempMax = min($this->numWorkers,count($this->queue->getQueued()));
            while(count($this->processes) < $tempMax) {
                $this->spawnWorker();
            }
        }
        if(empty($this->processes) && !empty($this->queued)) {
            throw new \Exception("Workers all died and were not replaced\n");
        }
        $this->queue->updateProgress();
    }

    public function spawnWorker() {
        $workerOptions = [];
        if(!is_null($this->workerBootstrap) && $this->bootstrapAsCallable===false) {
            if(is_array($this->workerBootstrap)) {
                $bsString = implode(" ",$this->workerBootstrap);
            } else {
                $bsString = $this->workerBootstrap;
            }
            $workerOptions[]=$bsString;
        }
        if(!is_null($this->logDir)) {
            $workerOptions[] = "log:".$this->logDir;
        }
        $workerString = empty($workerOptions) ? "" : implode(" ",$workerOptions);
        $process = new WorkerProcess("php -f ".__DIR__."/worker/Worker.php $workerString",[],$this->getProcessMeta());
        
        $process->setMetaData("workerTimeout",$this->workerTimeout);
        $this->processes[$process->getId()] = $process;
        $process->setMetaData("lastTime",microtime(true));
        $process->start();
        $this->addWorker($process);
    }

    /**
     * 
     * @param WorkerProcess $process 
     * @return mixed 
     */
    private function checkWorkerTimeout($process) {
        if(($lastTime = $process->getMetaData("lastTime",false)) !== false) {
            $pid = $process->getId();
            $workerTimeoutLimit = $this->workerTimeout;
            if(isset($this->processWork[$pid]) && isset($this->work[$this->processWork[$pid]]) && ($work = $this->queue->getWork($this->processWork[$pid])) !== false) {
                $workerTimeoutLimit = $work->getMetaData("timeout",$this->workerTimeout);
            }
            $elapsed = microtime(true)-$lastTime;
            if(!is_null($workerTimeoutLimit) && $workerTimeoutLimit > 0 && $elapsed>=$workerTimeoutLimit) {
                $this->logWorker($process,"worker timed out after",number_format($elapsed,3),"seconds");
                $this->processes[$pid]->stop();
                return true;
            }
        }
        return false;
    }
    
    private function removeWorker($pid) {
        unset($this->processes[$pid]);
        if(($k = $this->getWorkerMap($pid)) !== false) {
            unset($this->processMap[$k]);
        }
    }
    private function addWorker($process) {
        for($i=0;$i<$this->numWorkers;$i++) {
            if(!isset($this->processMap[$i])) {
                $this->processMap[$i] = $process->getId();
                return true;
            }
        }
        return false;
    }
    public function getWorkerMap($pid) {
        return array_search($pid,$this->processMap);
    }
    private function getProcessWorkCount(WorkerProcess $process,$inc=false) {
        if(!isset($this->processWorkCounts[$process->getId()])) {
            $this->processWorkCounts[$process->getId()]=0;
        }
        if($inc) {
            $this->processWorkCounts[$process->getId()]++;
        }
        return $this->processWorkCounts[$process->getId()];
    }
    private function checkWorkerWorkCount(WorkerProcess $process,$inc=false) {
        if($this->maxWorkerWork>0 && $this->getProcessWorkCount($process)>=$this->maxWorkerWork) {
            return false;
        }
        return true;
    }
    private function getProcessMeta() {
        return [
            "isMessageFrames"=>true,
            "useBuffer"=>false,
            "onStart"=>function(WorkerProcess $process) {
                dhGlobal::trace("[worker]",$process->getId(),":started:");
                $data = $process->Buffer()->read(true,false);
                $this->processOnStart($process);
            },
            "onDone"=>function(WorkerProcess $process,$exitCode=null,$termSignal=null) {
                dhGlobal::trace("[worker]",$process->getId(),":done:");
                $this->processOnDone($process,$exitCode,$termSignal);
            },
            "onData"=>function(WorkerProcess $process,$chunk=null) {
                $lines = explode("\n",$chunk);
                if(!empty($lines)) {
                    foreach($lines as $line) {
                        dhGlobal::trace("[worker]",$process->getId(),":data:",$line);
                        $this->processOnData($process,$line);
                    }
                }
            },
            "onError"=>function(WorkerProcess $process,$chunk=null) {
                $lines = explode("\n",$chunk);
                if(!empty($lines)) {
                    foreach($lines as $line) {
                        dhGlobal::trace("[worker-error]",$process->getId(),":error:",$line);
                        $this->processOnError($process,$line);
                    }
                } else {
                    dhGlobal::trace("[worker-error]",$process->getId(),":error:");
                    $this->processOnError($process);
                }
            },
        ];
    }
    private function processOnError(WorkerProcess $process,$chunk=null) {
        $this->logWorker($process,"<-- error:",$this->logDisplay($chunk)." ");
        if(!is_null($chunk)) {
            dhGlobal::error("[worker]",$process->getId(),$chunk);
        } else {
            dhGlobal::error("[worker]",$process->getId());
        }
    }
    private function processOnStreamError(WorkerProcess $process,$stream="",$chunk=null) {
        
    }
    private function processOnStart(WorkerProcess $process) {
        $this->logWorker($process,"onStart");
        $process->setMetaData("lastTime",microtime(true));
        $this->queue->updateWorkerStatus($process->getId(),"<ready>");
        if(!is_null($this->workerBootstrap) && $this->bootstrapAsCallable===true) {
            $this->sendBootstrapPacket($process);
        }
        dhGlobal::trace("[worker]",$process->getId(),"worker process started");
    }
    private function processOnDone(WorkerProcess $process,$exitCode=null,$termSignal=null) {
        $this->logWorker($process,"onDone");
        $process->setMetaData("lastTime",microtime(true));
        $this->queue->updateWorkerStatus($process->getId(),"<done>");
        dhGlobal::trace("[worker]",$process->getId(),"worker process exited");
        $this->removeWorker($process->getId());
        if(isset($this->processWork[$process->getId()])) {
            $workId = $this->processWork[$process->getId()];
            if(($work = $this->queue->getActive($workId)) !== false) {
                $this->workOnError($work,null,true);
            }
        }
        $this->checkWorkers(true);
        if(empty($this->processes)) {
            $this->queue->updateProgress(true);
        }
    }
    private function processOnData(WorkerProcess $process,$chunk=null) {
        $this->logWorker($process,"<--  data:",$this->logDisplay($chunk)." ");
        $process->setMetaData("lastTime",microtime(true));
        if(($message = Message::fromPacket($chunk)) !== false) {
            //dhGlobal::outLine("[worker]",$process->getPid(),"packet:",$message->getWorkId());
            if($message->getWorkId() == "init") {

                $this->queue->updateWorkerStatus($process->getId(),"<init>");
            } elseif($message->getWorkId() == "ack") {
            } elseif($message->getWorkId() == "ready") {
                $this->queue->updateWorkerStatus($process->getId(),"<ready>");
                if(!$this->checkWorkerWorkCount($process)) {
                    $this->sendNoWorkPacket($process);
                    return;
                }
                if(($work = $this->queue->next()) !== false) {
                    $this->sendProcessPacket($process,$work);
                    $this->processWork[$process->getId()] = $work->getId();
                    $this->queue->updateWorkerStatus($process->getId(),$work->getMetaData("name",$work->getId()));
                    $this->getProcessWorkCount($process,true);
                    return;
                } else {
                    //Maybe add a delay here to cut down on a lot of IO?
                    $this->sendNoWorkPacket($process);
                    return;
                }
            } elseif(($work = $this->queue->getActive($message->getWorkId())) !== false) {
                $process->setMetaData("status","<ready>");
                $this->queue->updateWorkerStatus($process->getId(),"<ready>");
                unset($this->processWork[$process->getId()]);
                if($message->isError()) {
                    $this->workOnError($work,$message);
                    if($this->killOnError) {
                        $process->stop();
                    }
                } else {
                    $this->workOnDone($work,$message);
                }
            }
        }
    }
    private function workOnDone(Work $work,$message=null) {
        return $this->queue->workOnDone($work,$message);
    }
    private function workOnError(Work $work,$message=null,$workerClosed=false) {
        return $this->queue->workOnError($work,$message,$workerClosed);
    }

    private function sendProcessPacket(WorkerProcess $process,Work $work) {
        $message = new Message();
        $message->setWorkId($work->getId());
        $message->setData($work->getWork());
        dhGlobal::trace("send","work",$message->toPacket());
        $process->write($message->toPacket()."\n");
        $this->logWorker($process,"-->","work",$message->toDisplay());
    }
    private function sendNoWorkPacket(WorkerProcess $process) {
        $message = new Message();
        $message->setWorkId("nowork");
        $message->setData(["time"=>microtime(true)]);
        dhGlobal::trace("send","noWork",$message->toPacket());
        $process->write($message->toPacket()."\n");
        $this->logWorker($process,"-->","noWork",$message->toDisplay());
    }
    private function sendBootstrapPacket(WorkerProcess $process) {
        $message = new Message();
        $message->setWorkId("init");
        $message->setData(["callable"=>["static","bootstrap"],"args"=>[$this->workerBootstrap]]);
        dhGlobal::trace("send","init",$message->toPacket());
        $process->write($message->toPacket()."\n");
        $this->logWorker($process,"-->","init",$message->toDisplay());
    }

    private function logWorker(WorkerProcess $process,...$args) {
        array_unshift($args,"[WORKER]",":".$process->getId().":");
        $this->_log(...$args);
    }
    private function _log(...$args) {
        if($this->log !== false) {
            if(is_null($this->logger)) {
                $this->logger = new dhOut(false,$this->log);
            }
            $this->logger->line(...$args);
        }
    }
    private function logDisplay($var,$len=150) {
        if(is_array($var)) {
            $var = json_encode($var);
        }
        if(!is_array($var) && !is_object($var)) {
            if(strlen($var)>$len) {
                $packet = substr($var,0,$len-4)." ...";
            }
        }
        return $var;
    }

    /**
	 * Get the value of numWorkers
     * @return  mixed
     */
    public function getNumWorkers() {
        return $this->numWorkers;
    }

    /**
     * Set the value of numWorkers
     * @param   mixed  $numWorkers  
     * @return  self
	 */
    public function setNumWorkers($numWorkers) {
        $this->numWorkers = $numWorkers;
        return $this;
    }

    /**
	 * Get the value of workerBootstrap
     * @return  mixed
     */
    public function getWorkerBootstrap() {
        return $this->workerBootstrap;
    }

    /**
     * Set the value of workerBootstrap
     * @param   mixed  $workerBootstrap  
     * @return  self
	 */
    public function setWorkerBootstrap($workerBootstrap) {
        $this->workerBootstrap = $workerBootstrap;
        return $this;
    }

    /**
	 * Get the value of bootstrapAsCallable
     * @return  mixed
     */
    public function getBootstrapAsCallable() {
        return $this->bootstrapAsCallable;
    }

    /**
     * Set the value of bootstrapAsCallable
     * @param   mixed  $bootstrapAsCallable  
     * @return  self
	 */
    public function setBootstrapAsCallable($bootstrapAsCallable) {
        $this->bootstrapAsCallable = $bootstrapAsCallable;
        return $this;
    }

    /**
	 * Get the value of workerErrorLimit
     * @return  mixed
     */
    public function getWorkerErrorLimit() {
        return $this->workerErrorLimit;
    }

    /**
     * Set the value of workerErrorLimit
     * @param   mixed  $workerErrorLimit  
     * @return  self
	 */
    public function setWorkerErrorLimit($workerErrorLimit) {
        $this->workerErrorLimit = $workerErrorLimit;
        return $this;
    }

    /**
	 * Get the value of maxWorkerWork
     * @return  mixed
     */
    public function getMaxWorkPerWorker() {
        return $this->maxWorkerWork;
    }

    /**
     * Set the value of maxWorkerWork
     * @param   mixed  $maxWorkerWork  
     * @return  self
	 */
    public function setMaxWorkPerWorker($maxWorkerWork) {
        $this->maxWorkerWork = $maxWorkerWork;
        return $this;
    }

    /**
	 * Get the value of workerTimeout
     * @return  mixed
     */
    public function getWorkerTimeout() {
        return $this->workerTimeout;
    }

    /**
     * Set the value of workerTimeout
     * @param   mixed  $workerTimeout  
     * @return  self
	 */
    public function setWorkerTimeout($workerTimeout) {
        $this->workerTimeout = $workerTimeout;
        return $this;
    }

    /**
	 * Get the value of killOnError
     * @return  mixed
     */
    public function getKillOnError() {
        return $this->killOnError;
    }

    /**
     * Set the value of killOnError
     * @param   mixed  $killOnError  
     * @return  self
	 */
    public function setKillOnError($killOnError) {
        $this->killOnError = $killOnError;
        return $this;
    }

    /**
	 * Get the value of log
     * @return  mixed
     */
    public function getLog() {
        return $this->log;
    }

    /**
     * Set the value of log
     * @param   mixed  $log  
     * @return  self
	 */
    public function setLog($log) {
        $this->log = $log;
        return $this;
    }

    /**
	 * Get the value of logger
     * @return  mixed
     */
    public function getLogger() {
        return $this->logger;
    }

    /**
     * Set the value of logger
     * @param   mixed  $logger  
     * @return  self
	 */
    public function setLogger($logger) {
        $this->logger = $logger;
        return $this;
    }

    /**
	 * Get the value of logDir
     * @return  mixed
     */
    public function getLogDir() {
        return $this->logDir;
    }

    /**
     * Set the value of logDir
     * @param   mixed  $logDir  
     * @return  self
	 */
    public function setLogDir($logDir) {
        if(substr($logDir,0,1) != "/") {
            $cwd = getcwd();
            if(substr($cwd,-1) != "/") {
                $cwd .="/";
            }
            $logDir = $cwd.$logDir;
        }
        $this->logDir = $logDir;
        return $this;
    }

    public function getProcessCount() {
        return count($this->processes);
    }
}