<?php
#namespace boru\dhutils\async;

use boru\dhutils\dhGlobal;
use boru\dhutils\async\Message;
use boru\dhutils\async\worker\WorkerException;
use boru\dhutils\async\worker\WorkerUtils;
use boru\dhutils\dhOut;
use React\Stream\ReadableResourceStream;
use React\Stream\WritableResourceStream;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;

$preload = false;
$preloadPackets = $includes = [];
$debug = false;
$maxWaitBeforeExit=5;
$logDir = null;
if(isset($argv) && is_array($argv) && !empty($argv)) {
    array_shift($argv);
    foreach($argv as $i=>$v) {
        if(substr($v,0,4) == "log:") {
            unset($argv[$i]);
            $t = explode(":",$v);
            if(isset($t[1])) { $logDir = $t[1]; }
        }
        if($v == "debug") {
            unset($argv[$i]);
            $debug=true;
            $maxWaitBeforeExit=0;
        } elseif($v == "preload") {
            unset($argv[$i]);
            $preload=true;
        } elseif(substr($v,0,1) == "#") {
            unset($argv[$i]);
            $preloadPackets[] = $v;
        } elseif(strpos($argv[$i],"/") !== false || strpos($argv[$i],".") !== false) {
            $includes[] = $argv[$i];
        }
    }
}
if(!$preload) {
    $preloadPackets = null;
}
if(!empty($includes)) {
    foreach($includes as $include) {
        if(file_exists($include)) {
            require_once $include;
        }
    }
}

$pparts = explode(DIRECTORY_SEPARATOR,__DIR__);
$pparts = array_reverse($pparts);

if($pparts[3] == "dhutils" && $pparts[5] == "vendor") {
    require_once __DIR__."/../../../../../autoload.php";
} else {
    require_once __DIR__."/../../../vendor/autoload.php";
}

/**
 * I understand completely that we shouldn't throw random ini_set definitions within libraries, but centos does weird things.
 */
if(php_sapi_name() == "cli") {
    ini_set("memory_limit",-1);
}
if(!is_null($logDir)) {
    initLogging($logDir);
}


$utilWorker = new Worker($preloadPackets,$debug,$maxWaitBeforeExit);

class Worker {
    private $id;
    private $delay = 0.1;
    private $maxWaitBeforeExit = 5;
    public static $bootstrap;
    private $debug = false;
    private $log = false;
    private $maxLogResponseLength=300;
    private $running = false;
    private $checking = false;
    private $terminated = false;

    private $readyLastSent;
    

    /** @var \React\Stream\ReadableResourceStream */
    private $stdin;
    /** @var \React\Stream\WritableResourceStream */
    private $stdout,$stderr;

    private $partialLine = "";

    public function __construct($preloadPackets=null,$debug=false,$maxWaitBeforeExit=5) {
        $this->id = uniqid();
        $this->maxWaitBeforeExit = $maxWaitBeforeExit;
        $this->debug = $debug;
        stream_set_blocking(STDIN, 0);
        if(!is_null($preloadPackets) && is_array($preloadPackets)) {
            foreach($preloadPackets as $packet) {
                $this->processPacket($packet);
            }
        }
        if(dhGlobal::get("asyncLog",false) !== false) {
            $this->log = true;
        }
        if(($ml = dhGlobal::get("asyncMaxLogResponseLength",false)) !== false) {
            $this->maxLogResponseLength = $ml;
        }

        $this->stdin = new ReadableResourceStream(STDIN);
        $this->stdout = new WritableResourceStream(STDOUT);
        $this->stderr = new WritableResourceStream(STDERR);

        $this->stdin    ->on("data", function($chunk) { $this->onInput("stdin",$chunk); });
        $this->stdout   ->on("data", function($chunk) { $this->onInput("stdout",$chunk); });
        $this->stderr   ->on("data", function($chunk) { $this->onInput("stderr",$chunk); });
        $this->stdin    ->on("close",function() { $this->onClose("stdin"); });
        $this->stdout   ->on("close",function() { $this->onClose("stdout"); });
        $this->stderr   ->on("close",function() { $this->onClose("stderr"); });
        $this->stdin    ->on("error",function(\Exception $e) { $this->onError("stdin",$e); });
        $this->stdout   ->on("error",function(\Exception $e) { $this->onError("stdout",$e); });
        $this->stderr   ->on("error",function(\Exception $e) { $this->onError("stderr",$e); });

        Loop::addPeriodicTimer($this->delay,function($timer) {
            if(!$this->running && !$this->checking) {
                $this->sendMessage($this->makeReadyMessage(),false);
                $this->checking=true;
                $this->readyLastSent = microtime(true);
            }
            if($this->maxWaitBeforeExit>0 && $this->checking && !is_null($this->readyLastSent) && microtime(true)-$this->readyLastSent>$this->maxWaitBeforeExit) {
                $elapsed = microtime(true)-$this->readyLastSent;
                $this->sendMessage($this->responseMessage("info",["what"=>"terminating","why"=>round($elapsed,2)." seconds elapsed since 'ready' was sent, exceeds limit of ".$this->maxWaitBeforeExit." seconds"]),false);
                $this->terminate(false);
            }
        });
        $this->log("init complete, started loop");
    }
    public function __destruct() {
        if(!$this->terminated) {
            $this->log("process exited abruptly",debug_backtrace());
        }
        $this->log("process exited");
    }
    
    private function onInput($streamType,$chunk) {
        $this->log("[".$streamType."]",trim($chunk));
        if($streamType == "stdin") {
            if(substr($chunk,-1) != "\n") {
                $this->partialLine.=$chunk;
            } else {
                $data = explode("\n",$this->partialLine.$chunk);
                $this->partialLine="";
                foreach($data as $line) { 
                    $this->processPacket($line);
                }
            }
        }
    }
    private function onClose($streamType) {
        $this->log("[".$streamType."]","---CLOSED---");
    }
    private function onError($streamType,\Exception $e) {
        $this->log("[".$streamType."]","---ERROR---",$e->getMessage());
    }

    public function processPacket($framedPacket) {
        //echo "processPacket: $framedPacket\n";
        if(($message = Message::fromPacket($framedPacket)) !== false) {
            $this->handlePacketMessage($message)->then(function($message) {
                $this->sendMessage($message);
            },function($message) {
                $this->sendMessage($message);
            });
        }
    }
    private function handlePacketMessage($message) {
        $packetDeferred = new Deferred();
        $this->running=true;
        $this->log(" <-- ",json_encode($message->get()));
        if(!$message->isValid()) {
            $packetDeferred->resolve($this->makeErrorMessage(0,Message::E_FRAME_ERROR,"unable to parse message frame"));
            
        } else {
            $workId = $message->getWorkId();
            $this->sendMessage($this->makeAckMessage($workId),false);

            if($workId == "nowork"){
                $this->processNoWork($workId,$message,$packetDeferred);

            } elseif($message->get("callable",false) !== false) {
                $this->processCallable($workId,$message,$packetDeferred);

            } else {
                $this->checking=false;
                $packetDeferred->resolve($this->makeErrorMessage($workId,Message::E_UNKNOWN_FRAME,$message->get()));

            }
        }
        return $packetDeferred->promise();
    }

    private function processNoWork($workId,$message,&$packetDeferred) {
        $this->checking=false;
        $packetDeferred->resolve(true);
        Loop::stop();
        $this->terminate();
        return $packetDeferred;
    }
    private function processCallable($workId,$message,&$packetDeferred) {
        if(($callable = $message->get("callable",false)) !== false) {
            $this->checking=false;
            $args = $message->get("args",[]);
            $asJson = $message->get("asJson",false);
            try {
                $executeResult = $this->execute($workId,$callable,$args,$asJson);
            } catch(\Exception $e) {
                $packetDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$e->getMessage(),$e->getTrace()));
            }
            if($executeResult instanceof Message) {
                $packetDeferred->resolve($executeResult);
            } elseif($executeResult instanceof PromiseInterface) {
                $executeResult->then(function($executeMessage) use ($packetDeferred) {
                    $packetDeferred->resolve($executeMessage);
                },function($e) use($workId,$packetDeferred) {
                    $packetDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$e->getMessage(),$e->getTrace()));
                });
            }
        } else {
            $packetDeferred->resolve(false);
        }
        return $packetDeferred;
    }



    private function execute($workId,$callable,$args,$asJson=false) {
        $this->logExecuteStart($callable);
        if(!is_array($args)) { $args = [$args]; }
        $result = $output = "";
        $trace = null;
        $success = false;
        if(!is_callable($callable)) {
            $this->logExecuteError($callable,Message::E_NOT_CALLABLE);
            $stringCallable = is_array($callable) ? implode("::",$callable) : $callable;
            return $this->makeErrorMessage($workId,Message::E_NOT_CALLABLE,"$stringCallable is not callable",null);
        }
        try {
            $result = call_user_func($callable,...$args);
            $success = true;
        } catch (WorkerException $e) {
            $this->logExecuteError($callable,$e->getMessage());
            return $this->makeErrorMessage($workId,$e->getType,$e->getMessage());
        } catch (\Exception $e) {
            $success=false;
            $result = $e->getMessage();
            $trace = $e->getTrace();
        }
        if($result instanceof \React\Promise\PromiseInterface) {
            return $this->handleExecDeferredResult($result,$workId,$success,$output,$callable,$asJson,$trace);
        } else {
            return $this->handleExecDirectResult($result,$workId,$success,$output,$callable,$asJson,$trace);
        }
    }
    private function handleExecDirectResult($result,$workId,$success,$output,$callable,$asJson,$trace) {
        $rawResult = null;
        if($success && $asJson && !is_array($result)) {
            $rawResult = $result;
            $result = json_decode($result,true);
        }
        if(!$success) {
            $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION." - ".$rawResult);
            return $this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$result,$trace);
        }
        $returnData = ["callable"=>$callable,"result"=>$result,"stdout"=>$output];
        if(!is_null($rawResult)) {
            $returnData["raw"]=$rawResult;
        }
        $this->logExecuteSuccess($callable);
        return $this->makeSuccessMessage($workId,$returnData);
    }
    private function handleExecDeferredResult($result,$workId,$success,$output,$callable,$asJson,$trace) {
        $execDeferred = new Deferred();
        $result->then(function($actualResult) use ($workId,$success,$output,$callable,$asJson,$trace,&$execDeferred) {
            $rawResult = null;
            if($success && $asJson && !is_array($actualResult)) {
                $rawResult = $actualResult;
                $actualResult = json_decode($actualResult,true);
            }
            if(!$success) {
                $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION." - ".$rawResult);
                $execDeferred->resolve($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$actualResult,$trace));
                return false;
            } else {
                $returnData = ["callable"=>$callable,"result"=>$actualResult,"stdout"=>$output];
                if(!is_null($rawResult)) {
                    $returnData["raw"]=$rawResult;
                }
                $this->logExecuteSuccess($callable);
                $execDeferred->resolve($this->makeSuccessMessage($workId,$returnData));
                
            }
        },function($e) use ($workId,$callable,&$execDeferred) {
            if(is_object($e) && method_exists($e,"getMessage")) {
                $result = $e->getMessage();
            } else {
                $result = $e;
            }
            if(is_object($e) && method_exists($e,"getTrace")) {
                $trace = $e->getTrace();
            } else {
                $trace = null;
            }
            $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION." - ".$result);
            $execDeferred->resolve($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$result,$trace));
            return false;
        });
        return $execDeferred->promise();
    }

    private function logExecuteStart($callable,$msg="") {
        $stringCallable = is_array($callable) ? implode("::",$callable) : $callable;
        $this->log("execute",$stringCallable,"START  ",$msg);
    }
    private function logExecuteSuccess($callable,$msg="") {
        $stringCallable = is_array($callable) ? implode("::",$callable) : $callable;
        $this->log("execute",$stringCallable,"SUCCESS",$msg);
    }
    private function logExecuteError($callable,$msg="") {
        $stringCallable = is_array($callable) ? implode("::",$callable) : $callable;
        $this->log("execute",$stringCallable,"ERROR  ",$msg);
    }

    public function makeSuccessMessage($workId,$data) {
        return $this->responseMessage($workId,$data);
    }
    public function makeErrorMessage($workId,$code,$message="",$trace=null) {
        return $this->responseMessage($workId,null,["code"=>$code,"message"=>$message,"trace"=>null]);
    }
    public function makeReadyMessage() {
        return $this->responseMessage("ready",["time"=>microtime(true)]);
    }
    public function makeAckMessage($workId=null) {
        return $this->responseMessage("ack",["workId"=>$workId,"time"=>microtime(true)]);
    }
    public function responseMessage($workId,$data=null,$error=null) {
        $message = new Message();
        $message->setWorkId($workId);
        $message->setData($data);
        $message->setError($error);
        return $message;
    }
    public function sendMessage(Message $message,$updateRunning=true) {
        /*if($this->maxLogResponseLength > 0) {
            $logmsg = substr(json_encode($message->get()),0,$this->maxLogResponseLength);
            if(strlen($logmsg)>=$this->maxLogResponseLength) {
                $logmsg.=" ...";
            }
        } else {
            $logmsg = json_encode($message->get());
        }
        */
        $this->log(" --> ",$message->toPacket());
        $this->stdout->write($message->toPacket()."\n");
        if($updateRunning) {
            $this->running=false;
        }
    }
    private function terminate($success=true) {
        $this->terminated=true;
        if(!$success) {
            exit(1);
        }
        exit();
    }

    private function log(...$args) {
        if($this->debug || $this->log) {
            array_unshift($args,"--".$this->id."--");
            dhGlobal::outLine(...$args);
        }
    }
    public static function bootstrap($bootstrap) {
        return WorkerUtils::bootstrap($bootstrap);
    }
    public static function testError(...$args) {
        return WorkerUtils::testError(...$args);
    }
    public static function exec(...$args) {
        return WorkerUtils::exec(...$args);
    }
    public static function http(...$args) {
        return WorkerUtils::http(...$args);
    }
}
function initLogging($logdir) {
    if(!is_dir($logdir)) {
        exec("mkdir -p ".$logdir);
    }
    $log_file = $logdir."/error.log";
    ini_set("log_errors",true);
    ini_set("dispaly_errors","off");
    ini_set('error_log', $log_file);

    dhGlobal::logger("debugger",dhGlobal::LOG_ALL,false,$logdir."/worker.log");
    dhGlobal::set("out",new dhOut(false,$logdir."/worker.log"));
    dhGlobal::set("asyncLog",true);
}