<?php
namespace boru\dhutils\async\worker;

use boru\dhutils\async\Message;
use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\parts\Buffer;
use boru\dhutils\queue\QueueableInterface;
use boru\dhutils\tools\ProcessQueue;
use React\Promise\Deferred;
use React\Promise\Promise;
use UnexpectedValueException;
use RuntimeException;

class WorkerProcess {

    private $command,$args,$meta,$priority;
    private $metaData;

    private $data;

    /** @var bool */
    private $running = false;
    /** @var bool */
    private $done = false;
    /** @var bool */
    private $ready = false;
    
    /** @var Buffer */
    private $buffer;
    private $useBuffer = true;
    private $isMessageFrames = false;
    private $messageBuffer;

    private $pid,$exitCode,$termSignal;

    /** @var callable */
    private $onStart,$onData,$onError,$onDone,$onStreamClose,$onStreamError;

    /** @var \React\Promise\Deferred */
    private $deferred;

    /** @var \React\ChildProcess\Process */
    private $process;

    public function __construct($command,$args=[],$meta=[]) {
        $this->setCommand($command);
        $this->setArgs($args);
        $this->setMeta($meta); //sets all the handlers/etc
        $this->setPid(uniqid());

        $this->buffer = new Buffer();
    }
    /**
     * Starts the Child Process, returns the Promise for easy chaining
     * @return Promise|false 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     */
    public function start() {
        $args = '';
        if(!empty($this->args)) {
            $args = ProcessQueue::pack($this->args);
        }
        $this->process = new \React\ChildProcess\Process('exec '.$this->command.' '.$args);
        $this->process->start();
        
        $this->setupProcessChannels();

        $this->onStart();
        return $this->Promise();
    }

    public function stop() {
        $this->running=false;
        $this->process->terminate();
    }

    
    public function terminate() {
        $this->running=false;
        $this->process->terminate();
        if(!is_null($this->deferred)) {
            $this->deferred->reject("terminated");
        }
    }

    public function get() {
        return [
            "pid"=>$this->pid,
            "running"=>$this->running,
            "done"=>$this->done,
            "meta"=>$this->meta,
            "args"=>$this->args,
            "command"=>$this->command,
            "buffer"=>$this->buffer,
            "exitCode"=>$this->exitCode,
            "termSignal"=>$this->termSignal,
            "data"=>$this->data,
        ];
    }

    public function write($data) {
        return $this->process->stdin->write($data);
    }

    public function isDone() {
        return $this->done;
    }
    public function isRunning() {
        return $this->running;
    }
    public function isReady() {
        return $this->ready;
    }

    /**
     * Return the processes's stdOut stream interface
     * @return \React\Stream\ReadableStreamInterface
     */
    public function stdOut() {
        return $this->process->stdout;
    }
    /**
     * Return the processes's stdErr stream interface
     * @return \React\Stream\ReadableStreamInterface
     */
    public function stdErr() {
        return $this->process->stderr;
    }
    /**
     * Return the processes's stdIn stream interface
     * @return \React\Stream\WritableStreamInterface
     */
    public function stdIn() {
        return $this->process->stdin;
    }

    /**
     * Get the buffer
     * @return Buffer 
     */
    public function Buffer() {
        return $this->buffer;
    }
    
    /**
     * Return the Deferred
     * @return Deferred 
     */
    public function Deferred() {
        return $this->deferred;
    }

    /**
     * Return the Promise (if exists)
     * @return Promise|false 
     */
    public function Promise() {
        return !is_null($this->deferred) ? $this->deferred->promise() : false;
    }

    /**
     * Wrapper for \React\Promise\Promise::then()
     * @param callable|null $onFulfilled 
     * @param callable|null $onRejected 
     * @return PromiseInterface 
     */
    public function then(callable $onFulfilled=null,callable $onRejected=null) {
        return $this->Promise()->then($onFulfilled,$onRejected);
    }



    /** Handler Methods */
    public function onDone($exitCode=null,$termSignal=null) {
        $this->done = true;
        $this->running = false;
        $this->ready = false;
        $this->exitCode = $exitCode;
        $this->termSignal = $termSignal;
        $this->deferred->resolve($this->buffer);
        if(!is_null($this->onDone)) {
            $handler = $this->onDone;
            $handler($this,$exitCode,$termSignal);
        }
    }
    public function onError($chunk=null) {
        if(!is_null($this->onError)) {
            $handler = $this->onError;
            $handler($this,$chunk);
        }
        if($this->useBuffer) {
            $this->buffer->err($chunk);
        }
    }
    public function onData($chunk=null) {
        if($this->isMessageFrames) {
            $data = !empty($this->messageBuffer) ? $this->messageBuffer : "";
            $data.=$chunk;
            if(($message = Message::fromPacket($data)) !== false) {
                $chunk = $data;
                $this->messageBuffer = null;
            } else {
                $this->messageBuffer = $data;
                return;
            }
        }
        if(!is_null($this->onData)) {
            $handler = $this->onData;
            $handler($this,$chunk);
        }
        if($this->useBuffer) {
            $this->buffer->out($chunk);
        }
    }
    public function onStart() {
        $this->running = true;
        $this->ready = false;
        if(!is_null($this->onStart)) {
            $handler = $this->onStart;
            $handler($this);
        }
    }
    public function onStreamClose($type) {
        if(!is_null($this->onStreamClose)) {
            $handler = $this->onStreamClose;
            $handler($this,$type);
        }
    }
    public function onStreamError($type,$chunk=null) {
        if(!is_null($this->onStreamError)) {
            $handler = $this->onStreamError;
            $handler($this,$type,$chunk);
        }
    }

    /**
     * Set the stdout/stderr channel listening
     * @return void 
     */
    private function setupProcessChannels() {
        $this->process->stdout->on('data', function($chunk) {
            $this->onData(trim($chunk));
        });
        $this->process->stdout->on('error', function($chunk) {
            $this->onStreamError("stdout",$chunk);
        });
        $this->process->stdout->on('close', function() {
            $this->onStreamClose("stderr");
        });
        $this->process->stderr->on('data', function($chunk) {
            $this->onError(trim($chunk));
        });
        $this->process->stderr->on('error', function($chunk) {
            $this->onStreamError("stderr",$chunk);
        });
        $this->process->stderr->on('close', function() {
            $this->onStreamClose("stderr");
        });
        $this->process->on('exit', function($exitCode,$termSignal) {
            $this->onDone();
        });
    }

    public function getId() {
        return $this->getPid();
    }
    /**
     * Get the value of pid
     * @return  mixed
     */
    public function getPid() {
        return $this->pid;
    }
    public function getPriority() {
        return $this->getPid();
    }

    /**
     * Set the value of pid
     * @param   mixed  $pid  
     * @return  self
     */
    public function setPid($pid) {
        $this->pid = $pid;
        return $this;
    }

    /**
     * Get the value of command
     * @return  mixed
     */
    public function getCommand() {
        return $this->command;
    }

    /**
     * Set the value of command
     * @param   mixed  $command  
     * @return  self
     */
    public function setCommand($command) {
        $this->command = $command;
        return $this;
    }

    /**
     * Get the value of args
     * @return  mixed
     */
    public function getArgs() {
        return $this->args;
    }

    /**
     * Set the value of args
     * @param   mixed  $args  
     * @return  self
     */
    public function setArgs($args) {
        $this->args = $args;
        return $this;
    }

    /**
     * Get the value of meta
     * @return  mixed
     */
    public function getMeta() {
        return $this->meta;
    }

    /**
     * Set the value of meta
     * @param   mixed  $meta  
     * @return  self
     */
    public function setMeta($meta) {
        $this->meta = $meta;
        $this->onData   = dhGlobal::getVal($meta, "onData",  null);
        $this->onError  = dhGlobal::getVal($meta, "onError", null);
        $this->onStart  = dhGlobal::getVal($meta, "onStart", null);
        $this->onDone   = dhGlobal::getVal($meta, "onDone",  null);
        $this->deferred = dhGlobal::getVal($meta, "deferred",null);
        $this->onStreamClose = dhGlobal::getVal($meta, "onStreamClose",null);
        $this->onStreamError = dhGlobal::getVal($meta, "onStreamError",null);
        $this->useBuffer = dhGlobal::getVal($meta, "useBuffer", true);
        $this->isMessageFrames = dhGlobal::getVal($meta, "isMessageFrames", false);
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
        }
        return $this;
    }

    /**
     * Get the value of running
     * @return  bool
     */
    public function getRunning() {
        return $this->running;
    }

    /**
     * Set the value of running
     * @param   bool  $running  
     * @return  self
     */
    public function setRunning($running) {
        $this->running = $running;
        return $this;
    }

    /**
     * Get the value of done
     * @return  bool
     */
    public function getDone() {
        return $this->done;
    }

    /**
     * Set the value of done
     * @param   bool  $done  
     * @return  self
     */
    public function setDone($done) {
        $this->done = $done;
        return $this;
    }

    /**
     * Set the Deferred that will be resolved on completion
     * @param Deferred $deferred 
     * @return $this 
     */
    public function setDeferred(Deferred $deferred) {
        $this->deferred = $deferred;
        return $this;
    }

    public function jsonSerialize($array=null) {
        if(is_null($array)) {
            $array = $this->get();
        }
        return $array;
    }
    public function __toString() {
        return json_encode($this);
    }

    public function setMetaData($key,$val="",$append=false) {
        if(strpos($key,".") !== false) {
            if($append) {
                $check = dhGlobal::getDot($this->metaData,$key);
                if(!is_null($check)) {
                    if(is_array($check)) {
                        $check[] = $val;
                        $val = $check;
                    } else {
                        $narr = [];
                        $narr[] = $check;
                        $narr[] = $val;
                        $val = $narr;
                    }
                }
            }
            dhGlobal::dotAssign($this->metaData,$key,$val);
        }
        else {
            if(isset($this->metaData[$key]) && $append) {
                if(is_array($this->metaData[$key])) {
                    $this->metaData[$key][] = $val;
                } else {
                    $temp = $this->metaData[$key];
                    $array[$key] = [];
                    $array[$key][] = $temp;
                    $array[$key][] = $val;
                }
            } else {
                $this->metaData[$key] = $val;
            }
        }
        return $this;
    }
    
    public function getMetaData($key=null,$default=null,$exists=false) {
        if(is_null($key)) {
            if($exists) {
                return !empty($this->metaData) ? true : false;
            } else {
                return !empty($this->metaData) ? $this->metaData : $default;
            }
        }
        if(strpos($key,".") !== false) {
            $uniqueid = uniqid("getArray",true);
            if(($check = dhGlobal::getDot($this->metaData,$key,$uniqueid)) !== $uniqueid) {
                return $exists ? true : $check;
            };
        }
        if($exists) {
            return isset($this->metaData[$key]);
        } else {
            return isset($this->metaData[$key]) ? $this->metaData[$key] : $default;
        }
        
    }

    public function metaDataExists($key) {
        return $this->getMetaData($key,null,true);
    }
}