<?php
namespace boru\dhutils;

use boru\dhcache\dhCache;
use boru\dhutils\dhOut;
use boru\dhutils\logger\Handler;
use boru\dhutils\tools\Template;
use boru\dhutils\traits\AsyncUtils;
use boru\dhutils\traits\GlobalDotParse;
use boru\dhutils\traits\GlobalFilesys;
use boru\dhutils\traits\GlobalLogHandler;
use boru\dhutils\traits\GlobalOut;
use boru\dhutils\traits\GlobalThreads;
use boru\dhutils\traits\GlobalUtils;

class dhGlobal {
    
    const TRIM_BOTH = 0;
    const TRIM_START = 1;
    const TRIM_END = 2;

    const LOG_ERROR = 1 << 0;
    const LOG_WARN = 1 << 1;
    const LOG_INFO = 1 << 2;
    const LOG_DEBUG = 1 << 3;
    const LOG_TRACE = 1 << 4;
    CONST LOG_ALL = 255;

    protected static $instance;
    protected static $globals = [];

    public static function add($name,$instance) {
        return static::set($name,$instance);
    }
    public static function set($name,$instance) {
        static::$globals[$name]=$instance;
        return static::get($name);
    }
    public static function remove($name) {
        if(isset(static::$globals[$name])) {
            unset(static::$globals[$name]);
        }
    }
    public static function get($name,$default=false) {
        if(!isset(static::$globals[$name])) {
            return $default;
        }
        return static::$globals[$name];
    }
    public static function globalIsset($name) {
        return isset(static::$globals[$name]);
    }
    public static function increment($var,$amount=1,$lockVar=false) {
        if($lockVar !== false) {
            while(isset(static::$globals[$lockVar])) {
                usleep(50000);
            }
        }
        static::$globals[$lockVar] = 1;
        $val = static::get($var,0);
        $val+=$amount;
        static::set($var,$val);
        if($lockVar !== false) {
            unset(static::$globals[$lockVar]);
        }
        return $val;
    }

    public static function setup($stdOut = true, $prefix = true, $outFile = false, $logFile = false) {
        $dhOut = new dhOut($stdOut, $outFile, $prefix);
        $logHandler = new Handler();
        static::add("out",$dhOut);
        static::add("logHandler",$logHandler);
    }

    /**
     * trait GlobalLogHandler
     * Wraps logHandler functions and makes them globally accessible
     */
    use GlobalLogHandler;

    /**
     * trait GlobalFilesys
     * Wraps Filesys related functions and makes them globally accessible
     */
    use GlobalFilesys;

    /**
     * trait GlobalDotParse
     * Wraps getDot and similar functions
     */
    use GlobalDotParse;
    use GlobalUtils;
    use GlobalOut;
    use GlobalThreads;
    use AsyncUtils;

    /**
     * 
     * @param array $options 
     * @return false|dhCache 
     */
    public static function cache($options=[]) {
        $cache = static::get("cache",false);
        if($cache === false) {
            $cache = new dhCache($options);
            static::add("cache",$cache);
            $cache = static::get("cache",false);
        }
        return $cache;
    }

    /**
     * Global access singleton for dhDB
     * 
     * Initiate or retrieve the dhDB connection object
     * 
     * @param array|null $config if not null, initiates the object
     * @param string $driver default="Mysql", the driver to use (TODO)
     * @return \boru\dhdb\dhDB
     */
    public static function db(array $config=null,$driver="Mysql") {
        $dbClass = "\\boru\\dhdb\\dhDB";
        if(class_exists($dbClass)) {
            if(!is_null($config)) {
                static::remove("dhdb");//in case it's already there, let's destroy it.
                $db = new $dbClass(["driver"=>$driver,"config"=>$config]);
                static::set("dhDB",$db);
            }
            $db = static::get("dhDB",false);
            if($db === false) {
                //try vtconfig.. maybe it exists
                if(static::fileIfExists("config.inc.php") !== false) {
                    $db = $dbClass::fromVtigerConfig("config.inc.php");
                    static::set("dhDB",$db);
                }
            }
            return static::get("dhDB",false);
        }
        throw new \Exception("cannot init DB without dhDB being included");
    }

    public static function dateIntervalToElapsed($dateInterval,$short=true,$includeZeros=false,$maxParts=false,$glue=" ") {
        $parts=[];
        if($dateInterval->y > 0 || $includeZeros) {
            $parts[] = $dateInterval->y. ($short?"Y":" ". static::pluralize($dateInterval->y,"year"));
        }
        if(($dateInterval->m > 0 || $includeZeros) && ($maxParts===false || count($parts)<$maxParts)) {
            $parts[] = $dateInterval->m. ($short?"M":" ". static::pluralize($dateInterval->m,"month"));
        }
        if(($dateInterval->d > 0 || $includeZeros) && ($maxParts===false || count($parts)<$maxParts)) {
            $parts[] = $dateInterval->d. ($short?"D":" ". static::pluralize($dateInterval->d,"day"));
        }
        if(($dateInterval->h > 0 || $includeZeros) && ($maxParts===false || count($parts)<$maxParts)) {
            $parts[] = $dateInterval->h. ($short?"h":" ". static::pluralize($dateInterval->h,"hour"));
        }
        if(($dateInterval->i > 0 || $includeZeros) && ($maxParts===false || count($parts)<$maxParts)) {
            $parts[] = $dateInterval->i. ($short?"m":" ". static::pluralize($dateInterval->i,"minute"));
        }
        if(($dateInterval->s > 0 || $includeZeros) && ($maxParts===false || count($parts)<$maxParts)) {
            $parts[] = $dateInterval->s. ($short?"s":" ". static::pluralize($dateInterval->s,"second"));
        }
        return implode($glue,$parts);
    }

    public static function pluralize($quantity, $singular, $plural=null) {
        if($quantity==1 || !strlen($singular)) return $singular;
        if($plural!==null) return $plural;
    
        $last_letter = strtolower($singular[strlen($singular)-1]);
        switch($last_letter) {
            case 'y':
                return substr($singular,0,-1).'ies';
            case 's':
                return $singular.'es';
            default:
                return $singular.'s';
        }
    }
    
    /**
     * 
     * @return Template 
     */
    public static function template() {
        if(($template = static::get("template",false)) !== false) {
            return $template;
        }
        return static::set("template",new Template());
    }
    public static function parseTemplate($templateString,$data=[]) {
        dhGlobal::template()->parse($templateString,$data);
    }

    /**
     * Get an array backtrace
     * @param int $levels 
     * @param int $skipLines 
     * @param mixed $traceArr 
     * @param bool $includeArgs 
     * @return array 
     */
    public static function backtrace($levels=9999,$skipLines=0,$traceArr=null,$includeArgs=false)
    {
        if (!function_exists('debug_backtrace')) return [];
        global $root_directory;

        if(is_null($traceArr)) {
            if($includeArgs) {
                $traceArr = debug_backtrace();
            } else {
                $traceArr = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS);
            }
        }
        array_shift($traceArr);
        array_shift($traceArr);
        $output = [];
        foreach ($traceArr as $arr) {
            if ($skipLines) {$skipLines -= 1; continue;}
            $levels -= 1;
            if ($levels < 0) break;

            $entry = [
                "line"=>"",
                "file"=>"",
                "func"=>"",
            ];
            $file = $arr['file'];
            if(!empty($root_directory) && substr($arr['file'],0,strlen($root_directory) == $root_directory)) {
                $file = substr($arr["file"],strlen($root_directory));
            }
            $func = "";
            if(isset($arr["class"])) {
                $func = $arr["class"];
                if(isset($arr["type"])) {
                    $func.=$arr["type"];
                }
            }
            if(isset($arr["function"])) {
                $func.=$arr["function"]."()";
            }
            $entry["line"] = $arr["line"];
            $entry["file"] = $file;
            $entry['func'] = $func;
            $output[] = $entry;
        }
    
        return $output;
    }
}