<?php
namespace boru\dhutils\filesys;

use boru\dhutils\dhGlobal;
use boru\dhutils\filesys\Directory;
use boru\dhutils\filesys\File;

class DirDiff {
    protected $new = [];
    protected $missing = [];
    protected $changed = [];
    protected $unchanged = [];

    protected $files;
    protected $metaFiles;

    protected $directory;

    public function __construct(Directory $directory,$meta=null) {
        $this->directory = $directory;
        $this->files = $directory->files();
        if(isset($this->files[$this->directory->getMetaFileName()])) {
            unset($this->files[$this->directory->getMetaFileName()]);
        }
        $this->diff($meta);
    }
    public function diff($meta=null) {
        if(is_null($meta)) {
            $meta = $this->directory->meta();
        }
        if(!$this->hasMetaData($meta)) {
            $this->new = array_keys($this->files);
        } else {
            $this->metaFiles = $meta["files"];

            foreach($this->files as $fileName=>$file) {
                if(!isset($this->metaFiles[$fileName])) {
                    $this->new[] = $fileName;
                }
            }
            foreach($this->metaFiles as $fileName=>$info) {
                if(!isset($this->files[$fileName])) {
                    $this->missing[] = $fileName;
                } elseif($info["sha1"] != $this->files[$fileName]->sha1()) {
                    $this->changed[] = $fileName;
                } else {
                    $this->unchanged[] = $fileName;
                }
            }
        }
        if(!empty($this->new)) {
            sort($this->new);
        }
        if(!empty($this->missing)) {
            sort($this->missing);
        }
        if(!empty($this->changed)) {
            sort($this->changed);
        }
        if(!empty($this->unchanged)) {
            sort($this->unchanged);
        }
    }
    private function hasMetaData($meta) {
        if(is_null($meta)
        || empty($meta)
        || (is_array($meta) && !isset($meta["files"]))
        || (is_array($meta) && empty($meta["files"]))) {
            return false;
        }
        return true;
    }

    /** @return File|array|false */
    public function getFile($fileName) {
        if(isset($this->files[$fileName])) {
            return $this->files[$fileName];
        } elseif(isset($this->metaFiles[$fileName])) {
            return $this->metaFiles[$fileName];
        }
        return false;
    }

    /** @return array|false */
    public function getNewOrChanged() {
        $merged = array_unique(array_merge(
            is_array($this->new) ? $this->new : [],
            is_array($this->changed) ? $this->changed : []
        ));
        sort($merged);
        return !empty($merged) ? $merged : false;
    }

    public function commitNew($pretty=false) {
        if($this->getNew() !== false) {
            foreach($this->getNew() as $fileName) {
                $this->directory->metaAdd($this->getFile($fileName));
            }
            $this->directory->metaCommit(false,$pretty);
            $this->diff();
        }
    }

    public function commitChanged($pretty=false) {
        if($this->getChanged() !== false) {
            foreach($this->getChanged() as $fileName) {
                $this->directory->metaAdd($this->getFile($fileName));
            }
            $this->directory->metaCommit(false,$pretty);
            $this->diff();
        }
    }

    public function commitMissing($pretty=false) {
        if($this->getMissing() !== false) {
            foreach($this->getMissing() as $fileName) {
                $this->directory->metaDel($fileName);
            }
            $this->directory->metaCommit(false,$pretty);
            $this->diff();
        }
    }

    public function commitAll($pretty=false) {
        $this->directory->metaCommit(true,$pretty);
        $this->diff();
    }

    /**
     * Get the value of new
     *
     * @return array|false
     */
    public function getNew() {
        return !empty($this->new) ? $this->new : false;
    }

    /**
     * Get the value of missing
     *
     * @return  array|false
     */
    public function getMissing() {
        return !empty($this->missing) ? $this->missing : false;
    }

    /**
     * Get the value of changed
     *
     * @return array|false
     */
    public function getChanged() {
        return !empty($this->changed) ? $this->changed : false;
    }

    /**
     * Get the value of unchanged
     *
     * @return array
     */
    public function getUnchanged() {
        return $this->unchanged;
    }
}