<?php
namespace boru\dhutils\filesys;

use boru\dhutils\base\dhObject;
use boru\dhutils\Dot;
use boru\dhutils\filesys\BasePath;
use boru\dhutils\FileUtil;

class Directory extends dhObject {
    protected $data = [
        "path"=>null,
        "name"=>null,
        "scanned"=>0
    ];
    protected $path;
    protected $name;
    protected $pathTrimmed;
    protected $separator = DIRECTORY_SEPARATOR;
    protected $files = [];
    protected $dirs = [];
    protected $expandSymlink = false;

    /** @var File */
    protected $metaFile;

    protected $meta;

    protected $metaFileName = ".bmeta";
    protected $metaPerms = "0600";

    protected $scanned = false;
    protected $create = false;

    /**
     * Valid options:
     * * path (default:current directory) path of the directory to load
     * * scan (default:true)
     * * recursive (default:false)
     * * expandSymlink (default:false)
     * * create (default:false) - create the directory (recursively) if it doesn't exist
     * * metaFile - sets the filename of the meta file (if meta is saved), default is .bmeta
     * * metaPerms - sets the file perms of the meta file (if meta is saved), default is 0600
     */
    public function __construct($options=[]) {
        $path = Dot::getVal($options,"path",null);
        $scan = Dot::getVal($options,"scan",true);
        $recursive = Dot::getVal($options,"recursive",false);
        $this->create = Dot::getVal($options,"create",false);
        $this->expandSymlink = Dot::getVal($options,"expandSymlink",false);
        if(($metaFileName = Dot::getVal($options,"metaFile",false)) !== false) {
            $this->metaFileName = $metaFileName;
        }
        if(($metaPerms = Dot::getVal($options,"metaPerms",false)) !== false) {
            $this->metaPerms = $metaPerms;
        }
        if(!is_null($path)) {
            $this->setDirectory($path);
        } else {
            $this->setDirectory(getcwd());
        }
        if($scan && is_dir($this->path())) {
            $this->scan($recursive);
        }
    }

    public function isScanned() {
        return $this->get("scanned",0) > 0 ? true : false;
    }

    /** @return DirDiff */
    public function metaDiff() {
        return new DirDiff($this,$this->meta());
    }

    /** Commits meta to file. If no meta exists, it is generated.
     * @param bool $all if true, overwrites existing meta with current file meta
     * @return array|null
     * */
    public function metaCommit($all=false,$pretty=true) {
        $meta = $this->meta();
        if(is_null($meta) || empty($meta) || $all) {
            $this->metaClear();
            if(!$this->isScanned()) {
                $this->scan();
            }
            foreach($this->files() as $file) {
                $meta["files"][$file->name()]=$file->meta();
            }
        }
        $meta["generated"] = date("U");
        $meta["path"] = $this->path();
        if(!isset($meta["files"])) {
            $meta["files"]=[];
        }
        if(isset($meta["files"][$this->metaFileName])) {
            unset($meta["files"][$this->metaFileName]);
        }
        ksort($meta["files"]);
        if($pretty) {
            $this->metaFile->write(json_encode($meta,JSON_PRETTY_PRINT));
        } else {
            $this->metaFile->write(json_encode($meta));
        }
        return $this->meta(true);
    }
    public function metaAdd($fileOrName) {
        $this->meta();
        if(($file = $this->getFileFromFileOrName($fileOrName)) !== false) {
            $this->meta["files"][$file->name()] = $file->meta();
        }
        return $this;
    }
    public function metaDel($fileOrName) {
        $this->meta();
        if(($fileName = $this->getFileFromFileOrName($fileOrName,true)) !== false) {
            if(isset($this->meta[$fileName])) {
                unset($this->meta[$fileName]);
            }
        }
        return $this;
    }
    protected function getFileFromFileOrName($fileOrName,$name=false) {
        if(!is_object($fileOrName)) {
            $file = FileUtil::fileIfExists($fileOrName);
            if($file !== false) {
                return $name ? $file->name() : $file;
            }
            return false;
        }
        return $name ? $fileOrName->name() : $fileOrName;
    }

    /** @return array|null */
    public function meta($reload=false) {
        if(is_null($this->metaFile)) {
            $this->metaFile = new File(["path"=>$this->path($this->metaFileName)]);
            $this->metaFile->perms($this->metaPerms);
        }
        if(is_null($this->meta) || $reload) {
            $this->metaClear();
            $this->meta = $this->metaFile->content(["json"=>true]);
            if(isset($this->meta["path"]) && $this->meta["path"] != $this->path()) {
                $this->metaClear();
            }
        }
        return $this->meta;
    }
    protected function metaClear() {
        $this->meta = ["generated"=>date("U"),"path"=>$this->path(),"files"=>[]];
    }

    public function path($fileOrDirName=null) {
        $path = $this->get("path",null);
        if(is_null($path)) {
            return null;
        }
        if(!is_null($fileOrDirName)) {
            return $path.$this->separator.$fileOrDirName;
        }
        return $path;
    }
    public function name() {
        return $this->get("name",null);
    }
    public function files($objects=true) {
        if($objects) {
            return $this->files;
        }
        $arr = [];
        foreach($this->files as $file) {
            $arr[$file->name()] = $file->get();
        }
        return $arr;
    }
    public function dirs($objects=true) {
        if($objects) {
            return $this->dirs;
        }
        $arr = [];
        foreach($this->dirs as $dir) {
            $arr[$dir->name()] = $dir->get();
        }
        return $arr;
    }
    public function file($name=null) {
        return $this->_getFile($name);
    }
    public function dir($name=null) {
        return $this->_getDir($name);
    }
    public function createFile($name,$touchOnExists=true) {
        $parts = explode($this->separator,$name);
        $name = array_pop($parts);
        $filename = $this->path().$this->separator.$name;
        if(file_exists($filename)) {
            if(!$touchOnExists) {
                throw new \Exception("File already exists ".$name,-1);
            } else {
                touch($filename);
                $file = new File([
                    "path"=>$filename,
                    "expandSymlink"=>$this->expandSymlink,
                ]);
            }
        } else {
            touch($filename);
            $file = new File([
                "path"=>$filename,
                "expandSymlink"=>$this->expandSymlink,
            ]);
            $this->_addFile($name,$file);
        }
        clearstatcache();
        return $file;
    }

    public function scan($recursive=false) {
        if($handle = opendir($this->path())) {
            while (false !== ($entry = readdir($handle))) {
                if($entry != "." && $entry != "..") {
                    if(is_dir($this->path($entry))) {
                        $dir = new Directory([
                            "path"=>$this->path($entry),
                            "scan"=>$recursive,
                            "recursive"=>$recursive,
                            "expandSymlink"=>$this->expandSymlink,
                        ]);
                        $this->_addDir($entry,$dir);
                    } else {
                        $file = new File([
                            "path"=>$this->path($entry),
                            "expandSymlink"=>$this->expandSymlink,
                        ]);
                        $this->_addFile($entry,$file);
                    }
                    
                }
            }
        }
        $size=0;
        foreach($this->files as $file) {
            $size+=!is_null($file->size()) ? $file->size() : 0;
        }
        $this->set("fileSize",$size);
        $this->set("scanned",time());
        return $this;
    }

    /**
     * 
     * @param array $findOptions 
     * @param array $outputOptions objects=true,fullPathName=false,callback=null
     * @return File[]|string[]|false 
     */
    public function find($findOptions=[],$outputOptions=[]) {
        $opts = ["depth","mindepth","maxdepth","noleaf","mount","xdev"];
        $tests = ["amin","atime","cmin","ctime","empty","false","fstype","gid","group","ilname","iname","inum","iwholename","iregex","links","lname","mmin","mtime","name","newer","newermt","nouser","not","nogroup","path","perm","regex","readable","writable","executable","wholename","size","true","type","uid","used","user","xtype","context"];
        $cmd = "find ".$this->path("/")." -maxdepth 1";
        $setOpts = ["maxdepth"=>1];
        $setTests = ["type"=>"f"];
        foreach($findOptions as $optName=>$optVal) {
            if(in_array($optName,$opts)) {
                $setOpts[$optName] = $optVal;
            } elseif(in_array($optName,$tests)) {
                $setTests[$optName] = $optVal;
            }
        }
        $parts = [];
        if(!empty($setOpts)) {
            foreach($setOpts as $optName=>$optVal) {
                if(is_array($optVal)) {
                    foreach($optVal as $optValVal) {
                        $parts[] = "-".$optName." ".$optValVal;
                    }
                } else {
                    $parts[] = "-".$optName." ".$optVal;
                }
            }
        }
        if(!empty($setTests)) {
            foreach($setTests as $optName=>$optVal) {
                if(is_array($optVal)) {
                    foreach($optVal as $optValVal) {
                        $parts[] = "-".$optName." ".$optValVal;
                    }
                } else {
                    $parts[] = "-".$optName." ".$optVal;
                }
            }
        }
        if(!empty($parts)) {
            $cmd.=" ".implode(" ",$parts);
        }
        $output = [];
        $files = [];
        //echo $cmd."\n";
        $r = exec($cmd,$output);

        $objects = Dot::getVal($outputOptions,"objects",true);
        $fullPathName = Dot::getVal($outputOptions,"fullPathName",false);

        if(!empty($output)) {
            foreach($output as $filePath) {
                if($objects) {
                    $file = new File(["path"=>$filePath]);
                    if($fullPathName) {
                        $files[$filePath] = $file;
                    } else {
                        $files[$file->name()] = $file;
                    }
                } else {
                    $files[] = $filePath;
                }
            }
        }
        return !empty($files) ? $files : false;
    }

    /** @return File|null */
    public function getMetaFile() {
        return $this->metaFile;
    }
    /** @return string */
    public function getMetaFileName() {
        return $this->metaFileName;
    }


    /**
     * Set the value of path
     *
     * @return  self
     */ 
    public function setDirectory($path)
    {
        $pathObj = new BasePath($path,$this->expandSymlink);
        if(!file_exists($pathObj->fullPath()) && $this->create)
        $pathObj->makeDir();
        $this->set("path",$pathObj->fullPath($this->expandSymlink));
        $this->set("pathObj",$pathObj);
        $pathTrimmed = substr($this->get("path"), strlen($this->separator));
        //$this->set("pathTrimmed",$pathTrimmed);
        $arr = explode($this->separator,$pathTrimmed);
        $this->set("name",array_pop($arr));
        return $this;
    }

    /**
     * Get the value of separator
     */ 
    public function getSeparator()
    {
        return $this->separator;
    }

    /**
     * Set the value of separator
     *
     * @return  self
     */ 
    public function setSeparator($separator)
    {
        $this->separator = $separator;

        return $this;
    }

    public function _addFile($name,$file=null) {
        if(is_null($file)) {
            $file = new File([
                "path"=>$this->path.$this->separator.$name
            ]);
        }
        $this->setArray($this->files,$name,$file,false);
        return $this;
    }
    public function _getFile($name=null) {
        return $this->getArray($this->files,$name,false);
    }
    public function _addDir($name,$dir=null) {
        if(is_null($dir)) {
            $dir = new Directory([
                "path"=>$this->path.$this->separator.$name
            ]);
        }
        $this->setArray($this->dirs,$name,$dir,false);
        return $this;
    }
    public function _getDir($name=null) {
        return $this->getArray($this->dirs,$name,false);
    }

    public static function fromPathString($pathString,$createIfNoExist=true,$options=[]) {
        if(!file_exists($pathString) && !$createIfNoExist) {
            return false;
        }
        if(!isset($options["create"])) {
            $options["create"] = $createIfNoExist;
        }
        $options["path"] = $pathString;
        return new self($options);
    }
    public static function fromInput($pathOrDirObject,$createIfNoExist=true,$options=[]) {
        if(is_object($pathOrDirObject) && $pathOrDirObject instanceof self) {
            return $pathOrDirObject;
        }
        return static::fromPathString($pathOrDirObject,$createIfNoExist,$options);
    }
}