<?php
namespace boru\dhutils\filesys;

use boru\dhutils\base\dhObject;
use boru\dhutils\dhGlobal;
use boru\dhutils\Dot;
use boru\dhutils\filesys\BasePath;

class File extends dhObject {
    protected $data = [
        "path"=>null,
        "name"=>null,
        "size"=>null,
        "type"=>null,
        "mtime"=>null,
        "ctime"=>null,
        "atime"=>null,
        "perms"=>null,
    ];

    protected $content;

    protected $stream;
    protected $separator = DIRECTORY_SEPARATOR;
    protected $expandSymlink = false;

    protected $create = true;
    protected $maxAge = false;
    protected $readMeta = true;

    /**
     * Opens a File Object instance on a file. This object provides easy access to read/write/meta information of a file.
     * 
     * Valid options
     * * string "path" - (required) - the path, including the filename, of hte file to load
     * * bool "readMeta" - (default:true) - pre-read basic meta information of the file if it exists
     * * bool "create" - (default:true) - create a file, including subdirs, if the path/filename does not exist
     * * bool "overwrite" - (default:false) - overwrite an existing file if it already exists, creating a new file
     * * string "content" - (optional) - Content to write to the file. If overwrite=false, it is appended to the file if the file exists.
     * * int "maxAge" - in seconds, if ctime is older than maxAge ago, replace the file
     */
    public function __construct($options=[]) {
        $path = Dot::getVal($options,"path",null);
        $this->readMeta = Dot::getVal($options,"readMeta",true);
        $this->create = Dot::getVal($options,"create",true);
        $this->maxAge = Dot::getVal($options,"maxAge",false);
        $overwrite = Dot::getVal($options,"overwrite",false);
        $content = Dot::getVal($options,"content",null);
        $this->expandSymlink = Dot::getVal($options,"expandSymlink",false);
        //$path="",$readMeta=true,$createIfNotExist=false
        if(!is_null($path)) {
            if(is_dir($path)) {
                throw new \Exception("$path is a directory and cannot be loaded as a File instance. Please use a Directory instance instead");
            }
            if(is_link($path) && !$this->expandSymlink) {
                //$path = 
            }
            $this->setFile($path);
            if(file_exists($path) && !is_dir($path)) {
                if($overwrite) {
                    if(!is_null($content)) {
                        $this->write($content,false);
                    } else {
                        if(($h = fopen($path,"w")) !== false) {
                            fclose($h);
                        }
                    }
                } else {
                    if(!is_null($content)) {
                        $this->write($content,true);
                    }
                }
                if($this->readMeta) {
                    $this->readMeta();
                }
            } else {
                throw new \Exception("$path does not exist.");
            }
        }
    }
    public function __destruct() {
        $this->streamClose();
    }

    public function path() {
        return $this->get("path",null);
    }
    public function name() {
        return $this->get("name",null);
    }
    public function dirPath() {
        return $this->get("dirPath",null);
    }
    public function type() {
        if(is_null($this->get("type",null))) {
            if(($type = filetype($this->path())) !== false) {
                $this->set("type",$type);
            }
        }
        return $this->get("type",null);
    }
    public function mimeType() {
        if(is_null($this->get("mimeType",null))) {
            if(($mimeType = mime_content_type($this->path())) !== false) {
                $this->set("mimeType",$mimeType);
            }
        }
        return $this->get("mimeType",null);
    }
    public function size() {
        if(is_null($this->get("size",null))) {
            if(($size = filesize($this->path())) !== false) {
                $this->set("size",$size);
            }
        }
        return $this->get("size",null);
    }
    public function mtime($timeFormat="timestamp") {
        if(is_null($this->get("mtime",null))) {
            if(($mtime = filemtime($this->path())) !== false) {
                $this->set("mtime",$mtime);
            }
        }
        return $this->timeFormat($this->get("mtime",null),$timeFormat);
    }
    public function ctime($timeFormat="timestamp") {
        if(is_null($this->get("ctime",null))) {
            if(($ctime = filectime($this->path())) !== false) {
                $this->set("ctime",$ctime);
            }
        }
        return $this->timeFormat($this->get("ctime",null),$timeFormat);
    }
    public function atime($timeFormat="timestamp") {
        if(is_null($this->get("atime",null))) {
            if(($atime = fileatime($this->path())) !== false) {
                $this->set("atime",$atime);
            }
        }
        return $this->timeFormat($this->get("atime",null),$timeFormat);
    }

    public function meta() {
        return [
            "mtime"=>$this->mtime("U"),
            "size"=>$this->size(),
            "sha1"=>$this->sha1(),
            "type"=>$this->type(),
        ];
    }
    /**
     * 
     * @param null|int if set, chmod's the file.. prefix with 0 (eg 0600 or 0777) 
     * @return int|null 
     */
    public function perms($newPerms=null) {
        if(!is_null($newPerms)) {
            chmod($this->path(),intval($newPerms, 8));
            clearstatcache(true,$this->path());
            if(($perms = fileperms($this->path())) !== false) {
                $perms = decoct($perms & 0777);
                $this->set("perms",$perms);
            }
        }
        if(is_null($this->get("perms",null))) {
            if(($perms = fileperms($this->path())) !== false) {
                $perms = decoct($perms & 0777);
                $this->set("perms",$perms);
            }
        }
        return $this->get("perms",null);
    }
    public function sha1() {
        if(is_null($this->get("sha1",null))) {
            if(($size = sha1_file($this->path())) !== false) {
                $this->set("sha1",$size);
            }
        }
        return $this->get("sha1",null);
    }
    public function sha1Content($useStream=false) {
        if(is_null($this->get("sha1",null))) {
            if($useStream) {
                $stream = $this->streamOpen("r");
                $context = hash_init("sha1");
                hash_update_stream($context,$stream);
                $hash = hash_final($context);
                $this->streamClose();
                $this->set("sha1",$hash);
            } else {
                $hash = sha1($this->content());
                $this->set("sha1",$hash);
            }
        }
        return $this->get("sha1");
    }
    public function delete($recreate=false) {
        unlink($this->path());
        $this->clearMeta();
        if($recreate) {
            touch($this->path());
            if($this->readMeta) {
                $this->readMeta();
            }
        }
    }
    
    public function streamOpen($mode="a+") {
        if(is_null($this->stream)) {
            $this->stream = fopen($this->path(),$mode);
        }
        return $this->stream;
    }
    public function streamClose() {
        if(!is_null($this->stream)) {
            fclose($this->stream);
        }
        $this->stream = null;
    }
    public function streamRead() {
        if(is_null($this->stream)) {
            throw new \Exception("Stream not open, use File::streamOpen()");
        }
        return fread($this->stream,$this->size());
    }
    public function streamWrite($content,$truncate=false) {
        if(is_null($this->stream)) {
            throw new \Exception("Stream not open, use File::streamOpen()");
        }
        if($truncate) {
            ftruncate($this->stream,0);
        }
        fwrite($this->stream,$content);
        fflush($this->stream);
        return $this;
    }
    public function lock($exclusive=true,$timeout=0) {
        if(is_null($this->stream)) {
            throw new \Exception("Stream not open, use File::streamOpen()");
        }
        if($timeout <= 0) {
            $lockMode = $exclusive ? LOCK_EX : LOCK_SH;
            flock($this->stream,$lockMode);
        } else {
            $time = microtime(true);
            $lockMode = $exclusive ? LOCK_EX | LOCK_NB : LOCK_SH | LOCK_NB;
            while(!flock($this->stream,$lockMode)) {
                usleep(round(rand(0, 100)*1000));
                if(microtime(true) - $time > $timeout) {
                    return false;
                }
            }
        }
        return true;
    }
    public function unlock() {
        if(is_null($this->stream)) {
            throw new \Exception("Stream not open, use File::streamOpen()");
        }
        if(!is_null($this->stream)) {
            flock($this->stream,LOCK_UN);
        }
        return true;
    }

    /**
     * Reads the contents of the file, passes it into $contentCallback, then writes the return from $contentCallback back into the file
     * @param callable $contentCallback 
     * @return string The content of the file after the operation has completed 
     */
    public function atomicReadWrite(callable $contentCallback,$timeout=10) {
        if(class_exists('\Memcached')) {
            $time = microtime(true);
            while(dhGlobal::cache()->lockVar($this->name()) === false) {
                usleep(round(rand(0, 100)*1000));
                if(microtime(true) - $time > $timeout) {
                    return false;
                }
            }
            $content = $contentCallback($this->content(["save"=>false]));
            $this->write($content);
            dhGlobal::cache()->unlockVar($this->name());
        } else {
            throw new \Exception("Please install Memcached");
            return false;
            if(!is_callable($contentCallback)) {
                throw new \Exception("contentCallback must be callable");
            }
            $this->streamClose();
            $this->streamOpen("r+");
            if($this->lock(true,$timeout)) {
                $content = $contentCallback($this->streamRead());
                rewind($this->stream);
                $this->streamWrite($content,true);
                $this->unlock();
                $this->streamClose();
                return true;
            }
            return false;
        }
    }
    /**
     * Retrieve the content of the file
     * 
     * Return the raw contents (default), or use options to transform it.
     * 
     * Options:
     * * json - boolean (default:false), parse the contents as json into an array.
     * * jsonObject - boolean (default:false), parse the contents as json into an object.
     * * filter - callable, filter/modify the content before returning it through json/jsonObject/plain
     * * save - boolean (default:true), save (cache) the raw content to the object.. not added to the get() output.
     */
    public function content($options=[]) {
        $json = Dot::getVal($options,"json",false);
        $jsonObject = Dot::getVal($options,"jsonObject",false);
        $lineFilter = Dot::getVal($options,"lineFilter",false);
        $lineEnding = Dot::getVal($options,"lineEnding",PHP_EOL);
        $filter = Dot::getVal($options,"filter",false);
        $save = Dot::getVal($options,"save",true);

        if(!is_null($this->content) && $save) {
            $content = $this->content;
        } else {
            $content = file_get_contents($this->path());
            if($save) {
                $this->content = $content;
            }
        }
        if(!is_null($lineFilter) && is_callable($lineFilter)) {
            $newContent = [];
            foreach(explode($lineEnding,$content) as $line) {
                if(($line = $lineFilter($line)) !== false) {
                    $newContent[] = $line;
                }
            }
            $content = implode($lineEnding,$newContent);
        }
        if(!is_null($filter) && is_callable($filter)) {
            $content = $filter($content);
        }
        if($json) {
            return json_decode($content,true);
        } elseif($jsonObject) {
            return json_decode($content);
        } else {
            return $content;
        }
    }
    public function write($content,$append=false) {
        if($append) {
            file_put_contents($this->path(),$content,FILE_APPEND);
        } else {
            file_put_contents($this->path(),$content);
        }
    }

    protected function timeFormat($timestampValue,$format="timestamp") {
        if(empty($timestampValue) || is_null($timestampValue) || $timestampValue <= 0) {
            return $timestampValue;
        }
        if(empty($format) || is_null($format)) {
            return $timestampValue;
        }
        try {
            $dt = new \DateTime(date("Y-m-d H:i:s",$timestampValue),new \DateTimeZone(date_default_timezone_get()));
        } catch (\Exception $e) {
            return $timestampValue;
        }
        if(!is_array($format)) {
            $objVals = ["datetime","dt","object"];
            if($format == "timestamp") {
                return $timestampValue;
            } elseif($format == "millis") {
                return $timestampValue*1000;
            } elseif(in_array(strtolower($format),$objVals)) {
                return $dt;
            } else {
                try {
                    $out = $dt->format($format);
                } catch (\Exception $e) {
                    $out = $timestampValue;
                }
                return $out;
            }
        } else {
            if(count($format) >= 2) {
                if(isset($format["format"]) && isset($format["timezone"])) {
                    $fmt = $format["format"];
                    $tz = $format["timezone"];
                } else {
                    $fmt = $format[0];
                    $tz = $format[1];
                }
                try {
                    $dt->setTimezone(new \DateTimeZone($tz));
                    $out = $dt->format($fmt);
                } catch (\Exception $e) {
                    $out = $timestampValue;
                }
                return $out;
            }
            return $timestampValue;
        }
        return $timestampValue;//shouldn't get here.. but just in case
    }

    /**
     * Set the value of path
     *
     * @return  self
     */ 
    public function setFile($path,$readMeta=true,$root=null)
    {
        $pathObj = new BasePath($path,$this->expandSymlink);
        if(!file_exists($path) && $this->create) {
            $pathObj->makeFile();
        }
        if(file_exists($path)) {
            $this->set("path",$pathObj->fullPath($this->expandSymlink));
        } else {
            throw new \Exception("File not found ".$path,-1);
        }

        if($this->maxAge !== false) {
            if($this->ctime() < time()-($this->maxAge)) {
                $this->delete(true);
            }
        }
        $pathTrimmed = substr($this->get("path"), strlen($this->separator));
        //$this->set("pathTrimmed",$pathTrimmed);
        $arr = explode($this->separator,$pathTrimmed);
        $this->set("name",array_pop($arr));
        $this->set("dirPath",$this->separator.implode($this->separator,$arr));
        return $this;
    }
    public function readMeta() {
        if(file_exists($this->path())) {
            $this->size();
            $this->type();
            $this->mtime();
            $this->atime();
            $this->ctime();
            $this->perms();
            $this->mimeType();
        }
        return $this;
    }
    public function clearMeta() {
        $this->set("size",null);
        $this->set("type",null);
        $this->set("mtime",null);
        $this->set("atime",null);
        $this->set("ctime",null);
        $this->set("perms",null);
        $this->set("mimeType",null);
    }
    public function reloadMeta() {
        $this->clearMeta();
        $this->readMeta();
    }

    public static function fromPathString($pathString,$createIfNoExist=true,$options=[]) {
        if(!file_exists($pathString) && !$createIfNoExist) {
            return false;
        }
        if(!isset($options["create"])) {
            $options["create"] = $createIfNoExist;
        }
        $options["path"] = $pathString;
        return new self($options);
    }
    public static function fromInput($pathOrFileObject,$createIfNoExist=true,$options=[]) {
        if(is_object($pathOrFileObject) && $pathOrFileObject instanceof self) {
            return $pathOrFileObject;
        }
        return static::fromPathString($pathOrFileObject,$createIfNoExist,$options);
    }
    public static function from($pathOrFileObject,$createIfNoExist=true,$options=[]) {
        return static::fromInput($pathOrFileObject,$createIfNoExist,$options);
    }
}