<?php
namespace boru\dhutils\logger;

use \boru\dhutils\dhOut;
use \boru\dhutils\dhGlobal;
use Exception;

class Handler extends \boru\dhutils\base\dhObject {
    public function __construct() {

    }
    protected $data = [];

    protected $logLevelsToBitMask = [
        "trace" => dhGlobal::LOG_TRACE,
        "debug" => dhGlobal::LOG_DEBUG,
        "info" => dhGlobal::LOG_INFO,
        "warn" => dhGlobal::LOG_WARN,
        "error" => dhGlobal::LOG_ERROR,
    ];
    protected $logLevelRanges = [
        "trace" => [401,500],
        "debug" => [301,400],
        "info"  => [201,300],
        "warn"  => [101,200],
        "error" => [1  ,100],
        "off"   => [0,0]
    ];
    protected $logLevels = [
        "trace" =>500,
        "debug" =>400,
        "info"  =>300,
        "warn"  =>200,
        "error" =>100,
        "off"   =>0,
    ];
    protected $prefixLength = 8;
    protected $prefixWrapper = ["[","]    "];
    protected $logPrefix = [
        "trace" => "TRACE",
        "debug" => "DEBUG",
        "info"  => "INFO",
        "warn"  => "WARN",
        "error" => "ERROR",
        "off"   => "",
    ];
    /**
     * 
     * @var dhOut[]
     */
    protected $loggers = [];
    protected $levelLoggers = [
        "trace"=>[],
        "debug"=>[],
        "info"=>[],
        "warn"=>[],
        "error"=>[]
    ];

    /**
     * Creates a new logger and sets it to the appropriate level
     * @param mixed $name -- name to identify this logger with
     * @param array $levels -- bitmask from dhGlobal::LOG_* constants
     * @param bool $stdOut -- obvious?
     * @param bool $file -- filename to enable output file logging, false to disabke
     * @param string $lineSeparator
     * @return void 
     * @throws Exception 
     */
    public function logger($name,$levels=[],$stdOut=true,$file=false,$lineSeparator=PHP_EOL) {
        if(empty($levels) || (!is_array($levels) && !is_int($levels))) {
            throw new \Exception("Logger levels must be defined and must be either an array of levels -- eg ['debug'] or ['error','warn'], OR use the dhGlobal::LOG_* constants as a bit mask -- eg LOG_ALL & ~LOG_TRACE");
        }
        $this->loggers[$name] = new dhOut($stdOut,$file,true,"date:Y-m-d H:i:s","\t",$lineSeparator);
        if(!is_array($levels) && is_int($levels)) {
            foreach($this->logLevelsToBitMask as $level=>$mask) {
                if($levels & $mask) {
                    $this->levelLoggers[$level][$name] = &$this->loggers[$name];
                }
            }
        }
        if(is_array($levels)) {
            foreach($levels as $level) {
                if(isset($this->logLevels[$level])) {
                    $this->levelLoggers[$level][$name] = &$this->loggers[$name];
                }
            }
        }
    }

    public function addLoggerToLevel($level,$loggerName) {
        if(!isset($this->loggers[$loggerName])) {
            throw new \Exception("logger $loggerName not initiated or found");
        }
        $label = $this->getLevelLabel($level);
        if(isset($this->levelLoggers[$label])) {
            
        }
    }

    
    public function log($level, ...$params) {
        $label = $this->getLevelLabel($level);
        $parent = $this->getParentLabel($level);
        $haveLoggers = false;
        if(isset($this->levelLoggers[$label]) && !empty($this->levelLoggers[$label])) {
            $haveLoggers = true;
        }
        if($parent != $label) {
            if(isset($this->levelLoggers[$parent]) && !empty($this->levelLoggers[$parent])) {
                $haveLoggers = true;
            }
        }
        if(!$haveLoggers) {
            return false;
        }
        $prefix = $this->getLevelPrefix($label);

        if(isset($this->levelLoggers[$label]) && !empty($this->levelLoggers[$label])) {
            foreach($this->levelLoggers[$label] as $logger) {
                if(!empty($params)) {
                    $logParams = $params;
                    array_unshift($logParams,$this->logPrefix($prefix));
                    call_user_func_array([$logger,"line"],$logParams);
                    unset($logParams);
                } else {
                    $logger->add("");
                }
                $logger->end();
            }
        }
        if($parent != $label && isset($this->levelLoggers[$parent]) && !empty($this->levelLoggers[$parent])) {
            foreach($this->levelLoggers[$parent] as $logger) {
                if(!empty($params)) {
                    $logParams = $params;
                    array_unshift($logParams,$this->logPrefix($prefix));
                    call_user_func_array([$logger,"line"],$logParams);
                    unset($logParams);
                } else {
                    $logger->add("");
                }
                $logger->end();
            }
        }
    }

    /**
     * define a custom log level.
     * 
     * Example:
     * 
     * $logger->defineLogLevel("query","debug","QRY1") -- adds a 'query' level to the debug log display
     * 
     * @param string $name the name of the level used for logging (eg: $logger->log($name,stuff,to,print))
     * @param string $level optional level to add the new level to, ['trace','debug','info','warn','error']
     * @param string $prefix optional prefix used if different from the name. 5 characters
     * @return boolean true if successful, false if failed
     */
    public function defineLogLevel($name,$level,$prefix=null) {
        if(is_null($prefix)) {
            $prefix = $name;
        }
        if(!is_numeric($level)) {
            $level = isset($this->logLevels[$level]) ? $level : "debug";
            list($start,$end) = $this->logLevelRanges[$level];
            $number = false;
            $levels = array_flip($this->logLevels);
            for($i=$start;$i<$end;$i++) {
                if(!isset($levels[$i])) {
                    $number = $i;
                    break;
                }
            }
            if($number === false) {
                return false;
            }
        } else {
            $number = $level;
        }
        $levels = $this->logLevels;
        $levels[$name] = $number;
        $temp = array_flip($levels);
        krsort($temp);
        $this->logLevels = array_flip($temp);

        $this->logPrefix[$name] = $prefix;
        return true;
    }

    public function logPrefix($prefix) {
        if(strlen($prefix)>$this->prefixLength) {
            $prefix = substr($prefix,0,$this->prefixLength);
        } elseif(strlen($prefix)<$this->prefixLength) {
            $prefix = str_pad($prefix,$this->prefixLength," ",STR_PAD_BOTH);
        }
        list($left,$right) = $this->prefixWrapper;
        return $left.strtoupper($prefix).$right;
    }

    public function getLogLevels() {
        return $this->logLevels;
    }
    public function getLogPrefixes() {
        return $this->logPrefix;
    }

    public function getLabelLevel($label) {
        return isset($this->logLevels[$label]) ? $this->logLevels[$label] : false;
    }
    public function getLevelPrefix($level) {
        $label = $this->getLevelLabel($level);
        if($label === false) {
            return false;
        }
        if(isset($this->logPrefix[$level])) {
            return $this->logPrefix[$level];
        }
        return false;
    }
    public function getLevelLabel($level) {
        if(!is_numeric($level) && isset($this->logLevels[$level])) {
            return $level;
        }
        if(is_numeric($level)) {
            $tflip = array_flip($this->logLevels);
            if(isset($tflip[$level])) {
                return $tflip[$level];
            }
        }
        return $this->getParentLabel($level);
    }
    public function getParentLabel($level) {
        if(!is_numeric($level) && isset($this->logLevels[$level])) {
            $level = $this->logLevels[$level];
        }
        if(is_numeric($level)) {
            foreach($this->logLevelRanges as $name=>$range) {
                if($level >= $range[0] && $level <= $range[1]) {
                    return $name;
                }
            }
        }
        return false;
    }


    public function trace(...$data) {
        array_unshift($data,"trace");
        call_user_func_array( [$this,"log"], $data);
    }
    public function debug(...$data) {
        array_unshift($data,"debug");
        call_user_func_array( [$this,"log"], $data);
    }
    public function info(...$data) {
        array_unshift($data,"info");
        call_user_func_array( [$this,"log"], $data);
    }
    public function warn(...$data) {
        array_unshift($data,"warn");
        call_user_func_array( [$this,"log"], $data);
    }
    public function error(...$data) {
        array_unshift($data,"error");
        call_user_func_array( [$this,"log"], $data);
    }




    protected static $instance;
    /**
     * Used to init the singleton with specific settings
     * @return dhlogger singlton
     */
    public static function init($level=null,$file=null,$dhOut=null) {
        if(self::$instance !== null) {
            self::instance()->end();
        }
        self::$instance = new self($level,$file,$dhOut);
        return self::$instance;
    }
    /**
     * Used to get the singleton isntance, or init with default settings;
     * @return dhlogger singlton
     */
    public static function instance() {
        if(self::$instance === null) {
            self::$instance = self::init();
        }
        return self::$instance;
    }

    /**
     * Get the value of prefixLength
     */ 
    public function getPrefixLength()
    {
        return $this->prefixLength;
    }

    /**
     * Set the value of prefixLength
     *
     * @return  self
     */ 
    public function setPrefixLength($prefixLength)
    {
        $this->prefixLength = $prefixLength;

        return $this;
    }

    /**
     * Get the value of prefixWrapper
     */ 
    public function getPrefixWrapper()
    {
        return $this->prefixWrapper;
    }

    /**
     * Set the value of prefixWrapper
     *
     * @return  self
     */ 
    public function setPrefixWrapper($prefixWrapper)
    {
        $this->prefixWrapper = $prefixWrapper;

        return $this;
    }
}