<?php
namespace boru\dhutils\multithread;

use \boru\dhutils\multithread\visualizer\BlankVisualizer;
use \boru\dhutils\dhGlobal;
use \boru\dhutils\dhThreads;

class Thread implements \JsonSerializable {
    protected $pid;
    protected $channelType;
    protected $command;
    protected $complete;
    protected $started;
    protected $meta = [];
    protected $stopped = false;
    protected $id;
    protected $runnerId;
    protected $wrapper = __DIR__."/../scripts/thread.php";
    protected $output;

    protected $json = false;
    protected $jsonObject = false;
    protected $callback;
    protected $timeout = 0;

    protected $debugMode = false;

    public function __construct($command,$meta=[],$options=[]) {
        $wrapper = dhGlobal::getVal($options,"wrapper",null);
        if(!is_null($wrapper)) {
            $this->wraper = $wrapper;
        }
        $this->json = dhGlobal::getVal($options,"json",false);
        $this->jsonObject = dhGlobal::getVal($options,"jsonObject",false);
        $this->channelType = dhGlobal::getVal($options,"channelType",dhThreads::CHANNEL_MEMCACHED);
        $this->id = dhGlobal::getVal($meta,"id",uniqid());
        $this->complete = false;
        $this->command = $command;
        $this->started = new \DateTime();
        $this->meta = $meta;
        $this->runnerId = uniqid("dhmt",true);

        if((dhGlobal::getVal($options,"debug",false)) !== false) {
            $this->debugMode = true;
        } elseif((dhGlobal::getVal($options,"debugMode",false)) !== false) {
            $this->debugMode = true;
        }

        $this->setupCallbacks($options);

        $this->execute();
    }
    public function __destruct() {
        $this->stop();
    }

    protected function execute() {
        $packet = $this->makePacket();
        $runCMD = "php -f ".$this->wrapper." ". dhThreads::pack($packet);
        $this->pid = exec($runCMD.' > /dev/null 2>&1 & echo $!');
        dhGlobal::trace("[MT-T] ".$runCMD);
        $this->debug(dhGlobal::padRight($this->pid,8," "),$runCMD);
    }

    protected function makePacket() {
        $packet = [
            "id"=>$this->runnerId,
            "command"=>$this->command,
            "return"=>true,
            "channelType"=>$this->channelType
        ];
        if($this->debugMode) {
            $packet["debugMode"] = true;
        }
        if($this->bootstrap !== false) {
            $packet["bootstrap"] = $this->bootstrap;
        }
        return $packet;
    }

    protected function setupCallbacks($options=[]) {
        $this->bootstrap = false;
        if(($bootstrapFile = dhGlobal::getVal($options,"bootstrap",false)) !== false) {
            if(file_exists($bootstrapFile)) {
                $this->bootstrap = $bootstrapFile;
            }
        }
        //default 'thread finished' callback that does not use a visualizer to process.
        $this->callback = dhGlobal::getVal($options,"callback",null);

    }

    /**
     * Terminates the thread
     */
    public function stop() {
        if(!$this->finished() && !$this->stopped) {
            $this->stopped = true;
            posix_kill($this->pid, 9);
            if($this->finished()) {
                dhGlobal::trace("[MT-T] ".$this->pid()." Stopped by request.");
            } else {
                dhGlobal::trace("[MT-T] ".$this->pid()." Recieved stop request.. kill signal sent");
            }
        }
    }

    public function collect() {
        $data = dhThreads::readChannel($this->runnerId,$this->channelType);
        if($data !== false) {
            if($this->json) {
                return json_decode($data,!$this->jsonObject);
            }
            return $data;
        }
        return false;
    }

    /**
     * Returns true if the thread has completed, false if running
     */
    public function finished() {
        if($this->complete) {
            return true;
        }
        if(!is_null($this->pid)) {
            if(!posix_getpgid($this->pid)) {
                $this->complete = true;
                $this->output = $this->collect();
                if(is_callable($this->callback)) {
                    $cb = $this->callback;
                    $cb($this);
                }
                return true;
            }
        }
        $now = time();
        if($this->timeout>0 && $this->started->format("U")+$this->timeout > $now) {
            $this->stop();
            return true;
        }
        $this->complete = false;
        return false;
    }

    /**
     * Returns true if the thread has completed, false if running
     */
    public function complete() {
        return $this->finished();
    }

    /**
     * Returns true if the thread has completed, false if running
     */
    public function isDone() {
        return $this->finished();
    }

    /**
     * Returns true if the thread has completed, false if running
     */
    public function done() {
        return $this->finished();
    }

    /**
     * Terminates the thread
     */
    public function kill() {
        $this->stop();
    }

    /**
     * Get the value of pid
     *
     * @return  mixed
     */
    public function pid() {
        return is_numeric($this->pid) ? $this->pid : false;
    }

    /**
     * Get the value of command
     *
     * @return  mixed
     */
    public function command() {
        return $this->command;
    }

    /**
     * Get the value of started
     *
     * @return  mixed
     */
    public function started() {
        return $this->started;
    }

    /**
     * Get the value of meta
     *
     * @return  mixed
     */
    public function meta() {
        return $this->meta;
    }

    /**
     * Get the value of output
     *
     * @return  mixed
     */
    public function output() {
        return $this->output;
    }

    /**
     * Get the value of runnerId
     *
     * @return  mixed
     */
    public function id() {
        return $this->id;
    }

    /**
     * Get the value of runnerId
     *
     * @return  mixed
     */
    public function runnerId() {
        return $this->runnerId;
    }

    public function get() {
        return [
            "pid"=>$this->pid(),
            "complete"=>$this->complete(),
            "command"=>$this->command(),
            "started"=>$this->started(),
            "meta"=>$this->meta(),
        ];
    }

    public function jsonSerialize($array=null) {
        if(is_null($array)) {
            $array = $this->get();
        }
        return $array;
    }
    public function __toString() {
        return json_encode($this);
    }

    public function debug(...$data) {
        if($this->debugMode) {
            array_unshift($data,"[T-child]  ");
            dhGlobal::debug(...$data);
        }
    }
}