<?php
namespace boru\dhuitls\util;

use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\parts\WorkerFrame;

/**
 * Packets are enclosed. How they are enclosed defines how they are parsed.
 * A packet is comprised of 4 elements, and terminated by a newline.
 * * START delimiter (#)
 * * TYPE definition
 * * TYPE delimiter ($)
 * * DATA
 * * END delimiter (#)
 * 
 * Method1: Plain Text json
 * * #plain${"callable":["\\boru\\dhutils\\dhGlobal","outLine"],"args":["arg1","arg2"...]}#
 * 
 * Method2: Base64 encoded json
 * * #base64$...encoded...#
 * 
 * Responses are returned in a json response, framed the same as it was received.
 * The response object consists of X elements
 * * callable
 * * startTime (unix timestamp)
 * * endTime (unix timestamp)
 * * output (string.. may be json encoded data) 
 */

$pparts = explode(DIRECTORY_SEPARATOR,__DIR__);
$pparts = array_reverse($pparts);
if($pparts[3] == "dhutils" && $pparts[5] == "vendor") {
    require __DIR__."/../../../../../autoload.php";
} else {
    require __DIR__."/../../../vendor/autoload.php";
}

$preload = false;
$preloadPackets = [];
$debug = false;
if(isset($argv) && is_array($argv) && !empty($argv)) {
    array_shift($argv);
    foreach($argv as $i=>$v) {
        if($v == "debug") {
            unset($argv[$i]);
            $debug=true;
        } elseif($v == "preload") {
            unset($argv[$i]);
            $preload=true;
        } elseif(substr($v,0,1) == "#") {
            unset($argv[$i]);
            $preloadPackets[] = $v;
        }
    }
}
if(!$preload) {
    $preloadPackets = null;
}
$utilWorker = new UtilWorker($preloadPackets,$debug);

class UtilWorker {
    public $delay = 0.1;

    public $bootstrap;

    private $lines = [];
    private $stop = false;

    private $debug = false;

    private $startTime;

    public function __construct($preloadPackets=null,$debug=false) {
        $this->debug = $debug;
        stream_set_blocking(STDIN, 0);
        if(!is_null($preloadPackets) && is_array($preloadPackets)) {
            foreach($preloadPackets as $packet) {
                $this->processLine($packet."\n");
            }
        }
        while(!$this->stop) {
            $this->loop();
            usleep($this->delay*1000000);
        }
    }
    public function loop() {
        while($line = fgets(STDIN)) {
            if(!empty($line)) {
                $this->processLine($line);
            }
        }
    }

    public function processLine($line) {
        if(substr($line,-1) != "\n") {
            $this->lines[] = $line;
        } else {
            $framedPacket = "";
            if(!empty($lines)) {
                $framedPacket = implode("",$this->lines);
            }
            $framedPacket.=trim($line);
            return $this->processPacket($framedPacket);
        }
    }
    public function processPacket($framedPacket) {
        $frame = WorkerFrame::fromPacket($framedPacket);
        $this->debug("[received]",json_encode($frame->get()));
        if(!$frame->valid()) {
            return $this->respondWithError("frameFailure","unable to parse message frame");
        }
        if(($bootstrap = $frame->get("bootstrap",false)) !== false) {
            return $this->bootstrap($bootstrap);
        } elseif(($callable = $frame->get("callable",false)) !== false) {
            $args = $frame->get("args",[]);
            $asJson = $frame->get("asJson",false);
            $workId = $frame->get("id",false);
            return $this->execute($callable,$args,$asJson,$workId);
        } else {
            print_r($frame->get());
        }
    }
    private function execute($callable,$args,$asJson=false,$workId) {
        if(!is_array($args)) { $args = [$args]; }
        $result = $output = "";
        $trace = null;
        $success = false;
        if(!is_callable($callable)) {
            $stringCallable = is_array($callable) ? implode("::",$callable) : $callable;
            return $this->respondWithError("not_callable","$stringCallable is not callable",null);
        }
        ob_start();
        try {
            $result = call_user_func($callable,...$args);
            $success = true;
        } catch (\Exception $e) {
            $success=false;
            $result = $e->getMessage();
            $trace = $e->getTrace();
            //return $this->respondWithError("execute_exception",$e->getMessage(),$e->getTrace());
        }
        $output = ob_get_contents();
        ob_end_clean();
        if($success && $asJson && !is_array($result)) {
            $result = json_decode($result,true);
        }
        if(!$success) {
            return $this->respondWithError("execute_exception",$result,$trace);
        }
        return $this->respondWithSuccess(["id"=>$workId,"callable"=>$callable,"data"=>$result,"stdout"=>$output]);
    }

    private function bootstrap($bootstrap) {
        if(!is_null($this->bootstrap)) {
            return $this->respondWithError("bootstrapped_already","already bootstrapped with ".json_encode($this->bootstrap));
        }
        $bootstrapFile = false;
        $setupCallable = false;
        $setupCallableArgs = [];
        if(!is_array($bootstrap)) {
            $bootstrapFile = $bootstrap;
        } elseif(isset($bootstrap["file"]) || isset($bootstrap["callable"])) {
            $bootstrapFile = isset($bootstrap["file"]) ? $bootstrap["file"] : false;
            $setupCallable = isset($bootstrap["callable"]) ? $bootstrap["callable"] : false;
            $setupCallableArgs = isset($bootstrap["args"]) ? $bootstrap["args"] : [];
        } else {
            return $this->respondWithError("bootstrap_invalid","Bootstrap must either be a filename or an array that includes 'file' and/or 'callable'");
        }
        if(!__bootstrap_include($bootstrapFile)) {
            return $this->respondWithError("bootstrap_not_found","File not found: ".$bootstrap);
        }
        if(($callableResponse = __bootstrap_call($setupCallable,$setupCallableArgs)) === false) {
            if(is_array($setupCallable)) { $setupCallable = implode("::",$setupCallable); }
            return $this->respondWithError("bootstrap_not_callable","Cannot call ".$setupCallable);
        }
        $this->bootstrap = $bootstrap;
        $output = [];
        if($bootstrapFile !== false) {
            $output["file"]=$bootstrapFile;
        }
        if($callableResponse !== false) {
            if($setupCallable !== false) {
                $output["callable"]=$setupCallable;
            }
            if($setupCallableArgs !== false) {
                $output["args"]=$setupCallableArgs;
            }
            $output["response"] = $callableResponse;
        }
        return $this->respondWithSuccess(["bootstrap"=>$this->bootstrap]);
    }

    public function respondWithSuccess($data) {
        $this->sendResponse(true,$data);
        return true;
    }
    public function respondWithError($code,$message="",$trace=null) {
        $this->sendResponse(false,["code"=>$code,"message"=>$message,"trace"=>$trace]);
        return false;
    }
    public function sendResponse($success=true,$data=null) {
        $output = [];
        $output["success"] = $success;
        if(!is_null($data)) {
            $output["data"] = $data;
        }
        $frame = WorkerFrame::fromData($output);
        $this->debug("[sending]",json_encode($frame->get()));
        echo $frame->getPacket()."\n";
    }
    public function terminate($success=true) {
        if(!$success) {
            exit(1);
        }
        exit();
    }

    private function debug(...$args) {
        if($this->debug) {
            dhGlobal::outLine(...$args);
        }
    }

    public static function exec(...$args) {
        if(!empty($args) && !is_null($args)) {
            $command = implode(" ",$args);
            passthru($command);
            return ["status"=>true,"command"=>$command];
        }
        throw new \Exception("static::exec requires at least 1 arg");
        return ["status"=>false];
    }
}

function __bootstrap_include($file) {
    if($file === false) {
        return true;
    }
    if(!file_exists($file)) {
        return false;
    }
    include $file;
    return true;
}
function __bootstrap_call($callable,$args) {
    if($callable === false) {
        return true;
    }
    if(!is_callable($callable)) {
        return false;
    }
    try {
        $return = call_user_func($callable,...$args);
    } catch (\Exception $e) {
        $return = false;
    }
    return $return;
}