<?php
namespace boru\dhutils\multithread\visualizer;

use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\MTCallback;
use boru\dhutils\multithread\MTCallbackInterface;
use boru\dhutils\traits\GetSetArray;

/**
 * Uses the dhOut outputter
 * 
 * @package boru\dhutils\multithread\visualizer
 */
class SimpleVisualizer extends MTCallback implements MTCallbackInterface {
    use getSetArray;
    public $threadStatus = [];
    public $threadMap = [];
    public $started = 0;
    public $done = 0;
    public $total = 0;

    protected $data = [];
    public $collectStats = [];
    public $statsCallback;

    /**
     * @var float
     */
    protected $startTime;

    /**
     * @var float
     */
    public $lastDisplayTime = 0;

    /**
     * Time in seconds between display prints
     * @var float
     */
    public $displayLoopTime = 0.5;

    public function __construct($threads=10,$totalWorkload=null,$statsCallback=null) {
        $this->total = !is_null($totalWorkload) ? $totalWorkload : $threads;
        $this->done = 0;
        for($i=0;$i<$threads;$i++) {
            $this->threadStatus[$i] = " ";
        }
        if(!is_null($statsCallback) && is_callable($statsCallback)) {
            $this->statsCallback = $statsCallback;
        }
    }

    /**
     * @param \boru\dhutils\dhThreads $pool the threadpool, so more work can be added if needed
     * @param \boru\dhutils\dhThreads\multithread\Thread $thread 
     * @return void 
     */
    public function threadStart($pool,$thread) {
        if(is_null($this->startTime)) {
            $this->startTime = microtime(true);
        }
        parent::threadStart($pool,$thread);
        $this->started++;
        $this->threadMap[$thread->runnerId()] = $this->nextIndex();
        $this->threadStatus[$this->threadMap[$thread->runnerId()]] = '.';


        if($this->started>$this->total) {
            $this->total = $this->started;
        }
    }

    /**
     * @param \boru\dhutils\dhThreads $pool the threadpool, so more work can be added if needed
     * @param \boru\dhutils\dhThreads\multithread\Thread $thread 
     * @return void 
     */
    public function threadFinished($pool,$thread) {
        parent::threadFinished($pool,$thread);
        $this->done++;
        $this->threadStatus[$this->threadMap[$thread->runnerId()]] = 'x';
        unset($this->threadMap[$thread->runnerId()]);
        $this->display();
    }

    /**
     * @param \boru\dhutils\dhThreads $pool
     * @return void 
     */
    public function loop($pool) {
        parent::loop($pool);
        $this->display();
    }

    /**
     * @param mixed $data
     * @return void 
     */
    public function collect($data) {
        parent::collect($data);
        if(is_callable($this->statsCallback)) {
            $cb = $this->statsCallback;
            $this->collectStats = $cb($data,$this);
        }
    }



    public function display() {
        if(!$this->canDisplay()) {
            return;
        }
        $extraStats="";
        if(!empty($this->collectStats)) {
            $arr = [];
            foreach($this->collectStats as $k=>$v) {
                $arr[] = "$k: $v";
            }
            $extraStats = implode(", ",$arr);
        }
        $this->lastDisplayTime = microtime(true);
        
        dhGlobal::outLine(implode(" ",$this->threadStatus)." ".$this->formatPercent(),$this->formatRuntime(),$extraStats);
    }
    protected function formatPercent() {
        $percent = (float) $this->done / (float) $this->total;
        $percent *= 100;
        $percent = "%".dhGlobal::pad(sprintf("%01.2f",$percent),6," ",STR_PAD_LEFT);
        return dhGlobal::pad($percent,12," ",STR_PAD_LEFT);
    }
    protected function formatRuntime() {
        $runTime = $this->runTime();
        if($runTime<=60) {
            $str = sprintf("%01.2f",$runTime)."s";
        } else {
            $runTime = round($runTime);
            $dtF = new \DateTime('@0');
            $dtT = new \DateTime("@$runTime");
            
            $str = dhGlobal::dateIntervalToElapsed($dtF->diff($dtT),true,false);
        }
        return "runTime: ".dhGlobal::pad($str,6," ",STR_PAD_LEFT);
    }

    protected function canDisplay() {
        if(microtime(true) - $this->lastDisplayTime >= $this->getDisplayLoopTime()) {
            return true;
        }
        return false;
    }
    /**
     * time elapsed since threads started
     * @return float 
     */
    protected function runTime() {
        return !is_null($this->startTime) ? microtime(true) - $this->startTime : 0;
    }

    protected function nextIndex() {
        foreach($this->threadStatus as $i=>$s) {
            if($s == ' ') {
                return $i;
            }
        }
        foreach($this->threadStatus as $i=>$s) {
            if($s == 'x') {
                return $i;
            }
        }
    }

    /**
     * Get time in seconds between display prints
     *
     * @return  float
     */
    public function getDisplayLoopTime() {
        return $this->displayLoopTime;
    }

    /**
     * Set time in seconds between display prints
     *
     * @param   float  $displayLoopTime  Time in seconds between display prints
     * @return  self
     */
    public function setDisplayLoopTime(float $displayLoopTime) {
        $this->displayLoopTime = $displayLoopTime;
        return $this;
    }
}