<?php
namespace boru\dhutils\tools;

use boru\dhutils\filesys\File;

class Output {
    protected static $instance;
    public $output;

    public $prefix=false;
    public $prefixFormat="date:Y-m-d H:i:s";
    public $prefixSpacer="\t";
    public $trailingNewLine=true;
    public $fileName=false;
    public $fileAppend=true;
    protected $file;
    public $stdOut=true;

    public $cursor = 0;
    public $indent = 0;

    public $lineSeparator;

    public $progressContent = null;
    public $progressStarted = false;

    private $lastTime = 0;


    /**
     * Used to init the singleton with specific settings
     * @return dhOut singlton
     */
    public static function init($stdOut=true,$fileName=false,$prefix=true,$prefixFormat="date:Y-m-d H:i:s",$prefixSpacer="\t",$lineSeparator=PHP_EOL) {
        if(self::$instance !== null) {
            self::instance()->end();
        }
        self::$instance = new self($stdOut,$fileName,$prefix,$prefixFormat,$prefixSpacer,$lineSeparator);
        return self::$instance;
    }
    /**
     * Used to get the singleton isntance, or init with default settings;
     * @return dhOut singlton
     */
    public static function instance() {
        if(self::$instance === null) {
            self::$instance = self::init();
        }
        return self::$instance;
    }

    public function __construct($stdOut=true,$fileName=false,$prefix=true,$prefixFormat="date:Y-m-d H:i:s",$prefixSpacer="\t",$lineSeparator=PHP_EOL) {
        $this->output = [];
        $this->setStdOut($stdOut);
        $this->setPrefix($prefix);
        $this->setPrefixFormat($prefixFormat);
        $this->setPrefixSpacer($prefixSpacer);
        $this->setFileName($fileName);
        $this->setLineSeparator($lineSeparator);
        $this->lastTime = -microtime(true);
    }
    public function __destruct() {
        if(!empty($this->output)) {
            $this->end();
        }
    }
    public function toOutput($content="") {
        $string = "";
        if(empty($content)) {
            $prevLine = $this->prevLine();
            if(!is_null($prevLine) && strlen($prevLine) == strlen(rtrim($this->prevLine(),$this->eol()))) {
                //$string.=$this->eol();
            }
            $string .= $this->currentLine();
        } else {
            $string = $content;
        }
        $string.=$this->eol();
        if($this->stdOut) {
            $commands = "";
            StdOut::output($commands.$string);
        }
        if(!is_null($this->file) && $this->fileName !== false) {
            $this->file->write($string,$this->fileAppend);
        }
    }
    public function clear() {
        $this->end();
    }

    public function a($data,$expectAppend=false) {
        return $this->add($data,$expectAppend);
    }
    public function line(...$args) {
        $parts = [];
        foreach($args as $arg) {
            $parts[] = $this->argToString($arg);
        }
        $line = $this->prefix();
        $line.= $this->indent();
        $line.= implode(" ",$parts);
        if($this->indent>0) {
            $line = str_replace($this->eol(),$this->eol().$this->indent(),$line);
        }
        $this->output[] = $line;
        $this->cursor++;
        $this->toOutput();
        return $this;
    }
    public function add($data,$expectAppend=false) {
        $line = $this->prefix();
        $line.= $this->indent();
        $line.= $this->argToString($data);
        if($this->indent>0) {
            $line = str_replace($this->eol(),$this->eol().$this->indent(),$line);
        }
        $this->output[] = $line;
        $this->cursor++;
        $this->toOutput();
        return $this;
    }

    protected function argToString($arg) {
        if(is_array($arg) || is_object($arg)) {
            return print_r($arg,true);
        } else {
            return $arg;
        }
    }

    public function merge(Output $otherOut) {
        if(!empty($this->output)) {
            $this->toOutput($this->eol());
        }
        $this->toOutput($otherOut->toString($this->eol(),false));
        $this->output = array_merge($this->output,$otherOut->output);
    }

    public function end() {
        $this->output = [];
        $this->cursor=0;
    }
    public function indent() {
        if($this->indent>0) {
            return str_pad("",$this->indent," ");
        }
        return "";
    }
    private function execTime() {
        $this->lastTime += microtime(true);
        $time = round($this->lastTime,4);
        $this->lastTime = -microtime(true);
        return $time;
    }
    public function prefix() {
        if($this->prefix) {
            if($this->prefixFormat == "execTime") {
                return "exec:{$this->execTime()}".$this->prefixSpacer;
            }
            $tt = explode(":",$this->prefixFormat,2);
            if(count($tt)==2) {
                if($tt[0] == "date") {
                    return date($tt[1]).$this->prefixSpacer;
                }
            }
            return $this->prefixFormat.$this->prefixSpacer;
        }
        return "";
    }
    public function toFile($filename,$flags=FILE_APPEND) {
        file_put_contents($filename,$this->toString(),$flags);
    }
    public function toString($lineSeparator=null,$trailing=true) {
        if(is_null($lineSeparator)) {
            $lineSeparator = $this->eol();
        }
        if($this->trailingNewLine && $trailing) {
            $o = $lineSeparator;
        } else {
            $o = "";
        }
        return implode($lineSeparator,$this->output).$o;
    }
    public function __toString() {
        return $this->toString($this->eol());
    }

    public function prevLine() {
        if($this->cursor>1) {
            return $this->output[$this->cursor-2];
        }
        return null;
    }
    public function currentLine() {
        if($this->cursor>0) {
            return $this->output[$this->cursor-1];
        }
        return null;
    }

    /**
     * Set the value of prefix
     *
     * @return  self
     */ 
    public function setPrefix($prefix)
    {
        $this->prefix = $prefix;

        return $this;
    }

    /**
     * Set the value of prefixFormat
     *
     * @return  self
     */ 
    public function setPrefixFormat($prefixFormat)
    {
        $this->prefixFormat = $prefixFormat;

        return $this;
    }

    /**
     * Set the value of trailingNewLine
     *
     * @return  self
     */ 
    public function setTrailingNewLine($trailingNewLine)
    {
        $this->trailingNewLine = $trailingNewLine;

        return $this;
    }

    /**
     * Set the value of fileName
     *
     * @return  self
     */ 
    public function setFileName($fileName)
    {
        if($fileName !== false) {
            $this->file = new File(["path"=>$fileName,"create"=>true]);
        }
        $this->fileName = $fileName;

        return $this;
    }

    /**
     * Set the value of stdOut
     *
     * @return  self
     */ 
    public function setStdOut($stdOut)
    {
        $this->stdOut = $stdOut;

        return $this;
    }

    /**
     * Set the value of prefixSpacer
     *
     * @return  self
     */ 
    public function setPrefixSpacer($prefixSpacer)
    {
        $this->prefixSpacer = $prefixSpacer;

        return $this;
    }

    /**
     * Set the value of fileAppend
     *
     * @return  self
     */ 
    public function setFileAppend($fileAppend)
    {
        $this->fileAppend = $fileAppend;

        return $this;
    }

    /**
     * Get the value of indent
     */ 
    public function getIndent()
    {
        return $this->indent;
    }

    /**
     * Set the value of indent
     *
     * @return  self
     */ 
    public function setIndent($indent)
    {
        $this->indent = $indent;

        return $this;
    }

    private function eol() {
        return $this->lineSeparator;
    }

    /**
     * Get the value of lineSeparator
     *
     * @return  mixed
     */
    public function getLineSeparator() {
        return $this->lineSeparator;
    }

    /**
     * Set the value of lineSeparator
     *
     * @param   mixed  $lineSeparator  
     * @return  self
     */
    public function setLineSeparator($lineSeparator) {
        $this->lineSeparator = $lineSeparator;
        return $this;
    }


    /**
     * 
     * @param string|string[] $char a string or array of strings with 'start', 'clear', and/or 'up' as values
     * @return string|void 
     */
    private function ansiChar($char="start|clear|up") {
        if(!is_array($char)) {
            if($char == "start") {
                return "\r";
            }
            if($char == "clear") {
                return "\033[K";
            }
            if($char == "up") {
                return "\033[1A";
            }
        } else {
            $chars = [];
            foreach($char as $c) {
                $chars[] = $this->ansiChar($c);
            }
            return implode("",$chars);
        }
    }
}