<?php
namespace boru\dhutils\tools;

use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\process\ClassProcess;
use boru\dhutils\multithread\process\Process;
use boru\dhutils\queue\Queue;
use boru\dhutils\queue\Visualizer;
use React\EventLoop\Loop;
use RuntimeException;

/**
 * A Queue for 1-shot processes.. launch the process, do the work, collect the results and throw it away.
 * 
 * @package boru\dhutils\tools
 */
class ProcessQueue {

    /** @var Queue */
    private $queue;

    private $max = 1;
    private $delay = 0.1;
    private $expected = 0;
    private $visualize = false;
    private $visualizerDelay = 1.0;

    /**
     * Variables for ClassProcess creation
     * @var mixed
     */
    private $callable,$bootstrap;

    public function __construct($max=1,$options=[],$queue=null,$visualizer=null) {
        $this->max = $max;
        $this->delay            = dhGlobal::getVal($options,"delay",$this->delay);
        $this->expected         = dhGlobal::getVal($options,"expected",$this->expected);
        $this->visualize        = dhGlobal::getVal($options,"visualize",false);
        $this->visualizerDelay  = dhGlobal::getVal($options,"visualizerDelay",$this->visualizerDelay);

        //ClassProcess shortcuts
        $this->callable         = dhGlobal::getVal($options,"callable",null);
        $this->bootstrap        = dhGlobal::getVal($options,"bootstrap",null);

        $this->initQueue($queue,$visualizer);
    }

    public function initQueue($queue=null,$visualizer=null) {
        if(is_null($queue)) {
            $this->queue = new Queue($this->max,$this->delay);
        } else {
            $this->queue = $queue;
        }
        if(method_exists($this->queue,"setVisualizer")) {
            if(is_null($visualizer)) {
                if($this->visualize) {
                    $visualizer = new Visualizer($this->max,$this->visualizerDelay,$this->expected);
                    $this->queue->setVisualizer($visualizer);
                }
            } else {
                $this->queue->setVisualizer($visualizer);
            }
        }
    }

    public function newClassProcess($args=null,$queue=false,callable $then=null) {
        $cp = new ClassProcess($this->callable,$args,$this->bootstrap);
        if(!is_null($then)) {
            $cp->then($then);
        }
        if($queue) {
            $this->queue($cp->Process(),true);
        }
        return $cp;
    }

    /**
     * 
     * @param mixed $command 
     * @param array $args 
     * @param array $meta 
     * @param bool $start 
     * @return Process 
     * @throws RuntimeException 
     */
    public function queueCommand($command,$args=[],$meta=[],$start=false) {
        $process = new Process($command,$args,$meta);
        return $this->queue($process,$start);
    }

    /**
     * 
     * @param Process $process 
     * @param bool $start 
     * @return Process 
     * @throws RuntimeException 
     */
    public function queue(Process $process,$start=false) {
        $this->queue->queue($process);
        if($start) {
            $this->queue->runOne();
        }
        return $process;
    }

    public function start($ticks=0) {
        $this->queue->run($ticks);
    }

    public function wait() {
        $this->start();
        Loop::run();
    }

    /**
     * 
     * @param callable|null $onResult 
     * @return array|true 
     */
    public function collect(callable $onResult=null) {
        $this->wait();
        $results = true;
        if(is_null($onResult)) {
            $results = [];
            $onResult = function(Process $process) {
                return $process->Buffer()->readOut();
            };
        }
        $finished = $this->queue->getDone();
        if(is_array($finished) && !empty($finished)) {
            foreach($finished as $process) {
                if(is_array($results)) {
                    $results[] = $onResult($process);
                } else {
                    $onResult($process);
                }
            }
        }
        return $results;
    }

    /**
     * Set the value of max
     *
     * @param   mixed  $max  
     * @return  self
     */
    public function setMax($max) {
        $this->max = $max;
        return $this;
    }

    /**
     * Set the value of delay
     *
     * @param   mixed  $delay  
     * @return  self
     */
    public function setDelay($delay) {
        $this->delay = $delay;
        return $this;
    }

    /**
     * Set the value of expected
     *
     * @param   mixed  $expected  
     * @return  self
     */
    public function setExpected($expected) {
        $this->expected = $expected;
        return $this;
    }

    /**
     * Set the value of visualize
     *
     * @param   mixed  $visualize  
     * @return  self
     */
    public function setVisualize($visualize) {
        $this->visualize = $visualize;
        return $this;
    }

    /**
     * Set the value of visualizerDelay
     *
     * @param   mixed  $visualizerDelay  
     * @return  self
     */
    public function setVisualizerDelay($visualizerDelay) {
        $this->visualizerDelay = $visualizerDelay;
        return $this;
    }

    /**
     * 
     * @param mixed $command 
     * @param array $args 
     * @param array $meta 
     * @return Process 
     */
    public static function Process($command,$args=[],$meta=[]) {
        $process = new Process($command,$args,$meta);
        return $process;
    }

    public static function ClassProcess($callable,$args=null,$bootstrap=null) {
        $process = new ClassProcess($callable,$args,$bootstrap);
        return $process;
    }

    public static function pack($data) {
        if(is_array($data) || is_object($data)) {
            $data = json_encode($data);
        }
        return base64_encode($data);
    }
    public static function unpack($packedData,$object=false) {
        $data = base64_decode($packedData);
        $test = json_decode($data,!$object);
        if(is_array($test) || is_object($test)) {
            return $test;
        }
        return $data;
    }
}