<?php
namespace boru\dhutils\traits;

use \boru\dhutils\dhGlobal;
use \boru\dhutils\dhThreads;

trait GlobalThreads {
     /**
     * Create or get a threadPool instance by name.
     * 
     * @param string|null $name 
     * @param int $numThreads 
     * @param float $throttleDelay 
     * @param bool $forceCreate 
     * @return dhThreads object
     */
    public static function threadPool($name=null,$numThreads=4,$throttleDelay=0.1,$forceCreate=false) {
        if(is_null($name)) {
            $name = "threadPool";
        }
        if(!static::globalIsset($name) || $forceCreate) {
            $threadPool = new dhThreads($numThreads,$throttleDelay);
            return static::set($name,$threadPool);
        }
        return static::get($name);
    }

    /**
     * Start a thread in the global threadPool instance
     * @param mixed $command 
     * @param array $meta 
     * @param array $options 
     * @return void 
     */
    public static function threadStart($command,$meta=[],$options=[]) {
        static::threadPool("threadPool")->start($command,$meta,$options);
    }

    /**
     * collect all current running threads from the global threadPool instance
     * @param mixed $collectType constant that defines the output of the collect() call valid options are:
     * * dhThreads::COLLECT_FULL -- each completed Thread object
     * * dhThreads::COLLECT_ARRAY -- array containing just the command output (if a valid dhCache instance is available to the environment (Memcached))
     * * dhThreads::COLLECT_NONE -- no output
     * @return array|true 
     */
    public static function threadCollect($collectType=null) {
        return static::threadPool("threadPool")->collect($collectType);
    }

    /**
     * Full-service wrapper for threading.
     * 
     * @param array $commands array of commands to execute in the thread pool. (command is either a string or an array of ["exec","data","meta","options"])
     * @param array $options options to apply to all commands executed {@see \boru\dhutils\dhThreads->start()}
     * @param int $numThreads number of threads to run
     * @param float $throttleDelay delay between waiting for available thread space
     * @return array array of Thread objects after completion
     * 
     * @see \boru\dhutils\dhThreads->start()
     */
    public static function threadMultiExec($commands=[],$options=[],$numThreads=4,$throttleDelay=0.1) {
        $pool = static::threadPool("threadPool",$numThreads,$throttleDelay);
        $collectOpts=[];
        $collectOpts["collectType"] = dhGlobal::getVal($options,"collectType",dhThreads::COLLECT_FULL);
        if(!is_array($commands)) {
            $commands = [$commands];
        }
        foreach($commands as $command) {
            $cmd = "";
            $meta = [];
            $opts = $options;
            if(is_array($command)) {
                if(($cmd = dhGlobal::getMultiVal($command,["command"=>false,"cmd"=>false,"exec"=>false],true)) !== false) {
                    $data = dhGlobal::getVal($command,"data",false);
                    $meta = dhGlobal::getVal($command,"meta",[]);
                    if($data !== false) {
                        $meta["data"] = $data;
                    }
                    foreach(dhGlobal::getVal($command,"options",[]) as $k=>$v) {
                        $opts[$k] = $v;
                    }
                } else {
                    $cmd = $command;
                }
            }
            $pool->start($cmd,$meta,$opts);
        }
        return $pool->collect($collectOpts);
    }
}