<?php
namespace boru\dhutils;

use boru\dhutils\multithread\MTCallbackInterface;
use \boru\dhutils\multithread\Thread;
use boru\dhutils\multithread\visualizer\BlankVisualizer;

class dhThreads {

    const CHANNEL_NONE = 0;
    const CHANNEL_MEMCACHED = 1;

    const COLLECT_NONE = -1;
    const COLLECT_FULL = 0;
    const COLLECT_OBJECT = 0;
    const COLLECT_ARRAY = 1;

    protected $maxThreads;
    protected $throttleDelay = 0.5;
    protected $waitTimeout = 0;
    protected $channelTimeout = 300;
    protected $channelType = 0;
    protected $waitingThreads = [];
    protected $runningThreads = [];
    public $finishedThreads = [];
    protected $stopped = false;
    protected $waiting = false;
    protected $started = false;
    protected $debugMode = false;

    protected $threadIndex = 0;

    /**
     * @var MTCallbackInterface
     */
    protected $callbackVisualizer;

    /**
     * Options:
     * * throttleDelay - float - delay in seconds to check threads
     * * channelType - const - defaults to static::CHANNEL_NONE
     */
    public function __construct($maxThreads=1,$throttleDelay=0.5,$options=[]) {
        $this->maxThreads = $maxThreads;
        $this->throttleDelay = $throttleDelay;
        $this->setOptions($options);
        declare(ticks = 1);
        pcntl_signal(SIGINT,[$this,"__shutdown"]);
    }
    public function __destruct() {
        $this->killAll();
    }
    public function __shutdown() {
        $this->__destruct();
        usleep(0.5*1000);
        exit();
    }
    public function killThread($thread) {
        if(is_object($thread)) {
            $thread->stop();
            return true;
        } elseif(is_numeric($thread)) {
            posix_kill($thread, 0);
            return true;
        } else {
            return false;
        }
    }
    public function killAll() {
        if(!$this->stopped) {
            $this->stopped = true;
            foreach($this->runningThreads as $thread) {
                $thread->stop();
            }
        }
    }

    public function startScript($file,$data=[],$options=[],$wait=true) {
        $meta = [];
        if(!empty($data)) {
            $args = $this->packData($data);
            $meta["data"] = $data;
        } else {
            $args = "";
            $meta = [];
        }
        if(is_object($file)) {
            $command = "php -f ".$file->path()." ".$args;
        } else {
            $command = "php -f ".$file." ".$args;
        }
        $this->start($command,$meta,$options,$wait);
    }
    public function start($command,$meta=[],$options=[],$wait=true) {
        if(!$this->stopped) { 
            $meta["id"] = $this->threadIndex;
            $this->threadIndex++;
            $thread = new Thread($command,$meta,$options);
            if($wait) {
                return $this->startThread($thread);
            } else {
                return $this->addThread($thread);
            }
        }
        return false;
    }
    protected function addThread($thread) {
        $this->waitingThreads[] = $thread;
        return $this;
    }
    protected function startThread($thread=null) {
        if(is_null($thread) && !empty($this->waitingThreads)) {
            $thread = array_shift($this->waitingThreads);
        }
        if(!is_null($thread)) {
            $this->throttle();
            if($this->callbackVisualizer instanceof MTCallbackInterface) {
                $this->callbackVisualizer->threadStart($this,$thread);
            }
            $this->runningThreads[] = $thread;
            dhGlobal::trace("[MT] Started thread with pid=".$thread->pid());
            $this->countRunningThreads();
            return $thread->pid();
        }
        return false;
    }

    public function collect($options=[]) {
        $collectType = dhGlobal::getVal($options,"collectType",null);
        $this->wait();
        if($collectType == static::COLLECT_ARRAY) {
            $return = [];
            foreach($this->finishedThreads as $thread) {
                $return[] = $thread->output();
            }
            return $return;
        } elseif($collectType == static::COLLECT_FULL || $collectType == static::COLLECT_OBJECT) {
            return $this->finishedThreads;
        }
        return true;
    }
    public function wait() {
        //dhGlobal::outLine(count($this->waitingThreads));
        $this->waiting=true;
        $this->throttle(1);
        while($this->startThread() !== false) {
        }
        $this->throttle(1);
        $this->waiting=false;
    }
    protected function throttle($maxThreads=null) {
        
        if(is_null($maxThreads)) {
            $maxThreads = $this->maxThreads;
        }
        if($maxThreads > 0) {
            while(($count = $this->countRunningThreads()) >= $maxThreads) {
                $this->throttled = true;
                dhGlobal::trace("[MT] $count runningThreads is >= $maxThreads.. waiting.",($this->throttleDelay * 1000000));
                usleep($this->throttleDelay * 1000000);
            }
            $this->throttled = false;
        }
    }

    protected function countRunningThreads() {
        foreach($this->runningThreads as $k=>$thread) {
            if($thread->finished()) {
                dhGlobal::trace("[MT] thread $k has completed");
                $this->finishedThreads[] = $thread;
                unset($this->runningThreads[$k]);
                if($this->callbackVisualizer instanceof MTCallbackInterface) {
                    $this->callbackVisualizer->collect($thread->output());
                }
                if($this->callbackVisualizer instanceof MTCallbackInterface) {
                    $this->callbackVisualizer->threadFinished($this,$thread);
                }
            }
        }
        if($this->callbackVisualizer instanceof MTCallbackInterface) {
            $this->callbackVisualizer->loop($this);
        }
        return count($this->runningThreads);
    }

    public function setOptions($options=[]) {
        //visualizer object.. default to BlankVisualizer.
        //visualizer holds 3 callbacks, threadStarted, threadFinished, and throttleWait (queued)
        $cbVisualizerClass = dhGlobal::getVal($options,"callbackVisualizer",new BlankVisualizer());
        if(!is_object($cbVisualizerClass) && !is_null($cbVisualizerClass)) {
            if(class_exists($cbVisualizerClass,true)) {
                $this->callbackVisualizer = new $cbVisualizerClass();
            }
        } elseif($cbVisualizerClass instanceof MTCallbackInterface) {
            $this->callbackVisualizer = $cbVisualizerClass;
        }
        if(is_null($this->callbackVisualizer)) {
            $this->callbackVisualizer = new BlankVisualizer();
        }

        if((dhGlobal::getVal($options,"debug",false)) !== false) {
            $this->debugMode = true;
        } elseif((dhGlobal::getVal($options,"debugMode",false)) !== false) {
            $this->debugMode = true;
        }

        if(($threadStartCallback = dhGlobal::getval($options,"threadStartCallback",false)) !== false) {
            $this->callbackVisualizer->setThreadStartCallback($threadStartCallback);
        }
        if(($threadFinishedCallback = dhGlobal::getval($options,"threadFinishedCallback",false)) !== false) {
            $this->callbackVisualizer->setThreadFinishedCallback($threadFinishedCallback);
        }
        if(($throttleWaitCallback = dhGlobal::getval($options,"throttleWaitCallback",false)) !== false) {
            $this->callbackVisualizer->setThrottleWaitCallback($throttleWaitCallback);
        }
    }

    public function debug(...$data) {
        if($this->debugMode) {
            array_unshift($data,"[T- main]  ");
            dhGlobal::debug(...$data);
        }
    }

    public function packData($data) {
        return static::pack($data);
    }
    public function unpackData($packedData,$object=false) {
        return static::unpack($packedData,$object);
    }
    public static function pack($data) {
        if(is_array($data) || is_object($data)) {
            $data = json_encode($data);
        }
        return base64_encode($data);
    }
    public static function unpack($packedData,$object=false) {
        $data = base64_decode($packedData);
        $test = json_decode($data,!$object);
        if(is_array($test) || is_object($test)) {
            return $test;
        }
        return $data;
    }

    public static function sendChannel($runnerID,$channelType,$data) {
        if($channelType == static::CHANNEL_MEMCACHED) {
            if(class_exists('\Memcached')) {
                return dhGlobal::cache()->lockSet($runnerID,$data);
            } else {
                dhGlobal::error("Memcached not work");
            }
        }
    }
    public static function readChannel($runnerID,$channelType) {
        if($channelType == static::CHANNEL_MEMCACHED) {
            if(class_exists('\Memcached')) {
                if(($data = dhGlobal::cache()->lockGet($runnerID))!==false) {
                    return $data;
                }
            } else {
                dhGlobal::error("Memcached not work");
            }
        }
    }
}