<?php
namespace boru\dhutils\filesys;

use \boru\dhutils\dhGlobal;
use boru\dhutils\StringUtil;

class BasePath {

    protected $fileName;
    protected $fullPath;
    protected $expandedPath;

    protected $baseDir;
    protected $expandedBaseDir;
    protected $remaining;
    
    protected $isDir = false;
    protected $isLink = false;

    private $basePathDebug = false;

    public function __construct($path=null,$expand=false,$debug=false) {
        if(!is_null($path)) {
            $this->resolvePath($path,$expand);
        }
        $this->basePathDebug = $debug;
    }

    public static function path($path,$expand=false,$debug=false) {
        $instance = new self($path,$expand,$debug);
        return $instance->fullPath($expand);
    }

    public function fullPath($expand=false) {
        return $expand ? $this->expandedPath : $this->fullPath;
    }
    public function baseDir($expand=false) {
        return $expand ? $this->expandedBaseDir : $this->baseDir;
    }
    public function isDir() {
        return $this->isDir();
    }
    public function isLink() {
        return $this->isLink();
    }
    public function makeDir() {
        mkdir($this->fullPath,0777,true);
        $this->resolvePath($this->fullPath);
    }
    public function makeFile() {
        if(!empty($this->remaining) && is_array($this->remaining)) {
            $this->createBase();
        }
        if(!file_exists($this->fullPath)) {
            touch($this->fullPath);
            $this->resolvePath($this->fullPath);
        }
    }
    public function createBase() {
        if(!empty($this->remaining) && is_array($this->remaining)) {
            $start = $this->baseDir;
            $dir = $start.DIRECTORY_SEPARATOR.implode(DIRECTORY_SEPARATOR,$this->remaining);
            if(!file_exists($dir)) {
                mkdir($dir,0777,true);
                $this->resolvePath($this->fullPath);
            }
        }
    }
    public function relative($testPath) {
        if(!is_object($testPath)) {
            $testPath = new BasePath($testPath);
        }
        $expandedParent = $testPath->fullPath(true);
        $fullPath = $this->fullPath(true);
        if(substr($fullPath,0,strlen($expandedParent)) !== $expandedParent) {
            return false;
        }
        $t = substr($fullPath,strlen($expandedParent));
        if(substr($t,0,strlen(DIRECTORY_SEPARATOR)) == DIRECTORY_SEPARATOR) {
            $t = substr($t,strlen(DIRECTORY_SEPARATOR));
        }
        return $t;
    }

    
    private function basePathDebug(...$args) {
        if($this->basePathDebug) {
            array_unshift($args,"debug");
            return dhGlobal::log(...$args);
        }
        return false;
    }
    protected function resolvePathExists($path,$expand=false) {
        if(substr($path,-1) == "/") {
            $path = substr($path,0,-1);
        }
        $parts = explode("/",$path);
        
        $this->basePathDebug("file exists");
        if(substr($path,0,1) !== "/") {
            $this->baseDir = is_link(getcwd()) ? readlink(getcwd()) : realpath(getcwd());
            $this->baseDir.= "/".dirname($path);
            $this->basePathDebug("relative",$this->baseDir);
        } else {
            $this->baseDir = dirname($path);
            $this->basePathDebug("real",$this->baseDir);
        }
        $this->expandedBaseDir = is_link($this->baseDir) ? readlink($this->baseDir) : realpath($this->baseDir);
        if(is_dir($path)) {
            $this->isDir = true;
        }
        $this->fileName = array_pop($parts);
        $this->fullPath = $this->baseDir;
        if(!is_null($this->fileName)) {
            $this->fullPath .= "/". $this->fileName;
        } else {
            $this->fullPath .= "/".array_pop($parts);
        }
        if(is_link($this->fullPath)) {
            $this->isLink = true;
            $this->expandedPath = readlink($this->fullPath);
        } else {
            $this->expandedPath = realpath($this->fullPath);
        }
        if(is_dir($this->expandedPath)) {
            $this->isDir = true;
        }
        $this->formatOutput();
        if($expand) {
            return $this->expandedPath;
        }
        return $this->fullPath;
    }
    private function resolvePathDots($path) {
        $parts = explode("/",$path);
        $newParts = [];
        while(!empty($parts)) {
            $part = array_shift($parts);
            if($part == "..") {
                if(count($newParts)>=1) {
                    array_pop($newParts);
                }
            } else {
                $newParts[] = $part;
            }
        }
        return implode("/",$newParts);
    }
    public function resolvePath($path,$expand=false) {
        $path = $this->toLinuxSlash($path);
        $this->basePathDebug("using $path");
        if(substr($path,-1) == "/") {
            $this->isDir = true;
            $path = substr($path,0,-1);
        }
        if(strpos($path,"..") !== false) {
            //resolve ../ first
            $this->basePathDebug("has ..");
            $path = $this->resolvePathDots($path,$expand);
        }
        $remaining = false;
        $parts = explode("/",$path);
        while(end($parts) == ".") {
            array_pop($parts);
            $parts = explode("/",implode("/",$parts));
        }
        if(empty($parts) || count($parts) === 1 && empty($parts[0])) {
            $path = getcwd();
            $parts = explode("/",$path);
        }
        if(file_exists($path)) {
            return $this->resolvePathExists($path,$expand);
        }

        $this->basePathDebug("file doesnt exist");
        if(substr($path,0,1) !== "/") {
            $this->baseDir = getcwd();
            $this->basePathDebug("relative ".$this->baseDir);
        } else {
            $this->baseDir = "/";
            $this->basePathDebug("real ".$this->baseDir);
        }
        $newParts = [];
        while(!empty($parts)) {
            $part = array_shift($parts);
            if(!empty($part)) {
                if(file_exists($this->baseDir."/".implode("/",$newParts)."/".$part)) {
                    $newParts[] = $part;
                } else {
                    array_unshift($parts,$part);
                    break;
                }
            }
        }
        $this->baseDir.= !empty($newParts) ? "/".implode("/",$newParts) : "";
        if(!empty($parts)) {
            $remaining = implode("/",$parts);
            $this->fileName = array_pop($parts);
        }
        unset($newParts);
        
        $this->basePathDebug("basedir= ".$this->baseDir);
        $this->baseDir = $this->baseDir;
        $this->expandedBaseDir = realpath($this->baseDir);
        if($remaining !== false) {
            $this->basePathDebug("remaining= $remaining");
            $this->remaining = $parts;
            $this->fullPath = $this->baseDir."/".$remaining;
            $this->expandedPath = realpath($this->baseDir);
            $this->expandedPath.="/".$remaining;
        } else {
            $this->fullPath = $this->baseDir;
            $this->expandedPath = realpath($this->baseDir);
        }
        

        $this->formatOutput();
        if($expand) {
            return $this->expandedPath;
        }
        return $this->fullPath;
    }
    protected function formatOutput() {
        $this->fullPath = !is_null($this->fullPath) ? $this->formatPathOut($this->fullPath) : null;
        $this->expandedPath = !is_null($this->expandedPath) ? $this->formatPathOut($this->expandedPath) : null;
        $this->baseDir = !is_null($this->baseDir) ? $this->formatPathOut($this->baseDir) : null;
        $this->expandedBaseDir = !is_null($this->expandedBaseDir) ? $this->formatPathOut($this->expandedBaseDir) : null;
        //$this->baseDir = !is_null($this->baseDir) ? $this->formatPathOut($this->baseDir) : null;
        //$this->baseDir = !is_null($this->baseDir) ? $this->formatPathOut($this->baseDir) : null;
    }
    private function formatPathOut($path) {
        return $this->toDirSeparator($this->toLinuxSlash($path));
    }
    private function toLinuxSlash($path) {
        
        //make all separators into / style separators
        if(DIRECTORY_SEPARATOR !== '/') {
            $path = str_replace(DIRECTORY_SEPARATOR, '/', $path);
        }
        $path = str_replace("\\","/",$path);
        while(strpos($path,"//") !== false) {
            $path = str_replace("//","/",$path);
        }
        //clean out useles '.' references
        $path = StringUtil::trimString("./",$path,StringUtil::TRIM_START);
        $path = StringUtil::trimString("/.",$path,StringUtil::TRIM_END);
        $path = str_replace('/./','/',$path);
        
        return $path;
    }
    private function toDirSeparator($path) {
        return str_replace("/",DIRECTORY_SEPARATOR,$path);
    }
}