<?php
namespace boru\dhutils\logger;

use \boru\dhutils\dhOut;

class Logger {
    protected $client;
    protected $file;

    /**
     * Room for custom debugging above error, 1-99
     * use dhLogger::addLogLevel() to add
     */
    protected $logLevels = [
        "trace" =>500,
        "debug" =>400,
        "info"  =>300,
        "warn"  =>200,
        "error" =>100,
        "off"   =>0,
    ];
    protected $logPrefix = [
        "trace" => "[TRACE]",
        "debug" => "[DEBUG]",
        "info"  => "[ INFO]",
        "warn"  => "[ WARN]",
        "error" => "[ERROR]",
        "off"   => "",
    ];
    protected $logLevel = 0;

    public function __construct($level=null,$file=null,$dhOut=null) {
        if(!is_null($dhOut)) {
            $this->client = $dhOut;
        } else {
            $this->client = new dhOut();
        }
        if(!is_null($level)) {
            $this->setLogLevel($level);
        }
        if(!is_null($file)) {
            $this->setFile($file);
        }
    }
    public function trace(...$data) {
        array_unshift($data,"trace");
        call_user_func_array( [$this,"log"], $data);
    }
    public function debug(...$data) {
        array_unshift($data,"debug");
        call_user_func_array( [$this,"log"], $data);
    }
    public function info(...$data) {
        array_unshift($data,"info");
        call_user_func_array( [$this,"log"], $data);
    }
    public function warn(...$data) {
        array_unshift($data,"warn");
        call_user_func_array( [$this,"log"], $data);
    }
    public function error(...$data) {
        array_unshift($data,"error");
        call_user_func_array( [$this,"log"], $data);
    }
    public function log($level, ...$params) {
        if(is_numeric($level)) {
            $ln = false;
            $temp = $this->logLevels;
            $tlevels = array_flip($temp);
            ksort($tlevels);
            foreach($tlevels as $lvl=>$nm) {
                if($lvl>=$level && $ln === false) {
                    $ln = $nm;
                }
            }
            if($ln !== false) {
                $prefix = $this->logPrefix[$ln];
            } else {
                $this->makeLogLevelPrefix($level);
            }
        } elseif(!is_numeric($level) && isset($this->logLevels[$level])) {
            $prefix = $this->logPrefix[$level];
            $level = $this->logLevels[$level];
        } else {
            $level = $this->logLevel;
            $prefix = $this->makeLogLevelPrefix($level);
        }
        if($this->logLevel>0 && $level<=$this->logLevel) {
            if(empty($params)) {
                $this->client->add("");
            } else {
                $this->client->add($prefix."   ");
                foreach($params as $param) {
                    $this->client->append(" ");
                    $this->client->append($param);
                }
            }
            $this->client->end();
        }
    }

    public function setFile($file) {
        $this->file = $file;
        $this->client->setFileName($file);
    }
    public function setLogLevel($level) {
        if($level === false) {
            $this->logLevel = 0;
        }
        if(is_numeric($level)) {
            $this->logLevel = $level;
        } elseif(isset($this->logLevels[strtolower($level)])) {
            $this->logLevel = $this->logLevels[strtolower($level)];
        }
    }
    public function level($level) {
        return $this->setLogLevel($level);
    }
    public function logLevel($level=null) {
        if(is_null($level)) {
            return $this->logLevel;
        }
        return $this->setLogLevel($level);
    }
    public function setLevel($level) {
        return $this->setLogLevel($level);
    }
    
    public function getMaxLevelForName($name) {
        $temp = $this->logLevels;
        $tlevels = array_flip($temp);
        ksort($tlevels);
        $max =0;
        $found = false;
        $level = $this->logLevels[$name];
        foreach($tlevels as $lvl=>$nm) {
            if($found) {
                $max = $lvl-1;
                break;
            } else {
                if($nm == $name) { 
                    $found = true;
                }
            }
        }
        if($max>0) { 
            return $max;
        } {
            return 9999;
        }
    }
    /**
     * Add a level to the logger.
     * 
     * Example:
     * 
     * $logger->addLogLevel("query","QUERY","debug") -- adds a 'query' level to the debug log display
     * 
     * @param string $name the name of the level used for logging (eg: $logger->log($name,stuff,to,print))
     * @param string $level optional level to add the new level to, ['trace','debug','info','warn','error']
     * @param string $prefix optional prefix used if different from the name. 5 characters
     * @return boolean true if successful, false if failed
     */
    public function addLogLevel($name,$level=null,$prefix=null) {
        if(is_null($prefix)) {
            $prefix = $name;
        }
        $prefix = $this->makeLogLevelPrefix($prefix);
        if(!is_numeric($level)) {
            $start = 999;
            $end = 0;
            if(!is_null($level)) {
                if(isset($this->logLevels[$level])) {
                    $start = $this->logLevels[$level];
                    $nextLevel = $this->getHigherLogLevel($level);
                    $end = $this->logLevels[$nextLevel];
                }
            }
            $number = false;
            $levels = array_flip($this->logLevels);
            for($i=$end+1;$i<$start;$i++) {
                if(!isset($levels[$i])) {
                    $number = $i;
                    break;
                }
            }
            if($number === false) {
                return false;
            }
        } else {
            $number = $level;
        }
        $levels = $this->logLevels;
        $levels[$name] = $number;
        $temp = array_flip($levels);
        krsort($temp);
        $this->logLevels = array_flip($temp);

        $this->logPrefix[$name] = $prefix;
        return true;
    }
    public function makeLogLevelPrefix($prefix) {
        if(strlen($prefix)>5) {
            $prefix = substr($prefix,0,5);
        } elseif(strlen($prefix)<5) {
            $prefix = str_pad($prefix,5," ",STR_PAD_BOTH);
        }
        return "[".strtoupper($prefix)."]";
    }

    public function getLogLevels() {
        return $this->logLevels;
    }
    public function getLogPrefixes() {
        return $this->logPrefix;
    }

    /**
     * Standard method wrapper
     */
    public function __call($method,$args) {
        return call_user_func_array( [$this->client,$method], $args);
    }

    protected static $instance;
    /**
     * Used to init the singleton with specific settings
     * @return dhlogger singlton
     */
    public static function init($level=null,$file=null,$dhOut=null) {
        if(self::$instance !== null) {
            self::instance()->end();
        }
        self::$instance = new self($level,$file,$dhOut);
        return self::$instance;
    }
    /**
     * Used to get the singleton isntance, or init with default settings;
     * @return dhlogger singlton
     */
    public static function instance() {
        if(self::$instance === null) {
            self::$instance = self::init();
        }
        return self::$instance;
    }

    public function getLabelForLevel($level=100) {
        $name = false;
        $temp = $this->logLevels;
        $tlevels = array_flip($temp);
        ksort($tlevels);
        foreach($tlevels as $lvl=>$lvlName) {
            if($lvl>=$level && $name === false) {
                $name = $lvlName;
                return $name;
            }
        }
    }
    public function getHigherLogLevel($level) {
        if(is_numeric($level)) {
            $level = $this->getLabelForLevel($level);
        }
        $keys = array_keys($this->logLevels);
        $current = array_search($level, $keys);
        $nextKey = $keys[($current===false ? -1 : $current)+1] ?: $keys[0];
        return $nextKey;
    }
}