<?php
namespace boru\dhutils\multithread\process;

use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\parts\WorkResult;
use boru\dhutils\queue\QueueableInterface;
use React\Promise\Deferred;

class Work implements QueueableInterface {

    /** @var bool */
    private $running = false;
    /** @var bool */
    private $done = false;
    /** @var bool */
    private $ready = false;

    private $id,$priority;
    
    /** @var WorkResult */
    private $result;

    private $callable,$args,$asJson;

    /** @var callable */
    private $onStart,$onData,$onError,$onDone;

    /** @var \React\Promise\Deferred */
    private $deferred;

    /**
     * 
     * @param callable $serializeableCallable either a string or an array, not a lambda function
     * @param mixed $args 
     * @param array $options 
     * @return void 
     */
    public function __construct($serializeableCallable,$options=[]) {
        if(is_object($serializeableCallable)) {
            throw new \Exception("serializeableCallable must be either a string or an array");
        }
        $this->id = uniqid();
        $this->callable  = $serializeableCallable;
        $this->args      = dhGlobal::getVal($options, "args",     null);
        $this->asJson    = dhGlobal::getVal($options, "asJson",   false);
        $this->onData    = dhGlobal::getVal($options, "onData",   null);
        $this->onError   = dhGlobal::getVal($options, "onError",  null);
        $this->onStart   = dhGlobal::getVal($options, "onStart",  null);
        $this->onDone    = dhGlobal::getVal($options, "onDone",   null);
        $this->deferred  = dhGlobal::getVal($options, "deferred", null);
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
        }
    }

    public function getId() {
        return $this->id;
    }
    public function getPriority() {
        return $this->priority;
    }

    public function isDone() {
        return $this->done;
    }
    public function isRunning() {
        return $this->running;
    }
    public function isReady() {
        return $this->ready;
    }

    public function start() { 
        $this->onStart();
    }
    public function done($result) {
        $this->result = $result;
        $this->onDone();
    }
    public function error($data=null) {
        $this->onError($data);
    }

    public function terminate() { }

    public function getWork() {
        $packet = [];
        $packet["id"]       = $this->id;
        $packet["callable"] = $this->callable;
        $packet["args"]     = $this->args;
        $packet["asJson"]   = $this->asJson;
        return $packet;
    }

    /**
     * Get the value of result
     *
     * @return  mixed
     */
    public function getResult() {
        return $this->result;
    }

    /** Handler Methods */
    public function onDone() {
        $this->done = true;
        $this->running = false;
        $this->ready = false;
        $this->deferred->resolve($this->result);
        if(!is_null($this->onDone)) {
            $handler = $this->onDone;
            $handler($this,$this->result);
        }
    }
    public function onError($chunk=null) {
        if(!is_null($this->onError)) {
            $handler = $this->onError;
            $handler($this,$chunk);
        }
        $this->deferred->reject($chunk);
        $this->buffer->err($chunk);
    }
    public function onData($chunk=null) {
        if(!is_null($this->onData)) {
            $handler = $this->onData;
            $handler($this,$chunk);
        }
        $this->buffer->out($chunk);
    }
    public function onStart() {
        $this->running = true;
        $this->ready = false;
        if(!is_null($this->onStart)) {
            $handler = $this->onStart;
            $handler($this);
        }
    }

    /**
     * Get the value of running
     * @return  bool
     */
    public function getRunning() {
        return $this->running;
    }

    /**
     * Set the value of running
     * @param   bool  $running  
     * @return  self
     */
    public function setRunning($running) {
        $this->running = $running;
        return $this;
    }

    /**
     * Get the value of done
     * @return  bool
     */
    public function getDone() {
        return $this->done;
    }

    /**
     * Set the value of done
     * @param   bool  $done  
     * @return  self
     */
    public function setDone($done) {
        $this->done = $done;
        return $this;
    }

    /**
     * Set the Deferred that will be resolved on completion
     * @param Deferred $deferred 
     * @return $this 
     */
    public function setDeferred(Deferred $deferred) {
        $this->deferred = $deferred;
        return $this;
    }

    /**
     * Get the value of deferred
     *
     * @return  mixed
     */
    public function getDeferred() {
        return $this->deferred;
    }

    /** @return \React\Promise\Promise */
    public function Promise() {
        return $this->deferred->promise();
    }
    /** @return \boru\dhutils\multithread\process\Process */
    public function Process() {
        return $this->process;
    }

    /**
     * Wrapper for \React\Promise\Promise::then()
     * @param callable|null $onFulfilled 
     * @param callable|null $onRejected 
     * @return \React\Promise\PromiseInterface
     */
    public function then(callable $onFulfilled=null,callable $onRejected=null) {
        return $this->Promise()->then($onFulfilled,$onRejected);
    }

    public function run($options=[]) {
        $worker = dhGlobal::getVal($options,"worker",null);
        $bootstrap = dhGlobal::getVal($options,"bootstrap",null);
        if(($debugPackets = dhGlobal::getVal($options,"debugPackets",false)) === false) {
            $debugPackets = dhGlobal::getVal($options,"debug",false);
        }
        if(is_null($worker)) {
            $worker = new Worker($bootstrap,null,$debugPackets);
        }
        $worker->startWork($this);
        $this->then(function($result) use ($worker) {
            $worker->stop();
        });
        return $this;
    }
}