<?php
namespace boru\dhutils\multithread\process;

use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\parts\WorkerFrame;
use boru\dhutils\multithread\parts\WorkResult;
use boru\dhutils\tools\WorkQueue;
use React\Promise\Deferred;

class Worker {

    private $id;

    /** @var WorkQueue */
    private $queue;

    /** @var Process */
    private $process;

    /** @var Work */
    private $work;

    /** @var bool */
    private $running = false;
    /** @var bool */
    private $ready = false;
    /** @var bool */
    private $done = false;

    private $bootstrap;
    private $bootstrapped = false;

    /** @var \React\Promise\Deferred */
    private $internalDeferred;

    /** @var \React\Promise\Deferred */
    private $onDoneDeferred;

    private $debugPackets = false;


    public function __construct($bootstrap=null,WorkQueue $queue=null,$debugPackets=false) {
        $this->id = uniqid();
        $this->bootstrap = $bootstrap;
        $this->queue=$queue;
        $this->debugPackets = $debugPackets;
        
        $this->process = new Process("php -f ".__DIR__."/../util/Worker.php",[],$this->getProcessMeta());
    }

    public function isDone() {
        return $this->done;
    }
    public function isRunning() {
        return $this->running;
    }
    public function isReady() {
        return $this->ready;
    }

    public function start() {
        if(!$this->process->isRunning()) {
            $this->process->start();
        }
    }
    public function stop() {
        $this->process->stop();
    }

    private function onStart() {
        $this->ready=true;
        $this->running=false;
        $this->done=false;
        $this->onDoneDeferred = new Deferred();
        if(!is_null($this->bootstrap)) {
            $this->sendPacket(["bootstrap"=>$this->bootstrap]);
        }
    }
    private function onData(Process $process,$chunk=null) {
        $this->processPacket($chunk);
    }
    private function onDone(Process $process,$exitCode=null,$termSignal=null) {
        $this->running = false;
        $this->ready = false;
        $this->done = true;
        $this->onDoneDeferred->resolve($exitCode);
    }

    private function sendPacket($data) {
        $frame = WorkerFrame::fromData($data);
        $packet = $frame->getPacket();
        $this->packetDebug("sent",$packet,$data);
        
        $this->process->write($packet."\n");
    }

    public function startWork(Work $work) {
        if(!$this->process->isRunning()) {
            $this->process->start();
        }
        $this->ready=false;
        $this->running=true;
        $this->done=false;
        $this->internalDeferred = new Deferred();
        $promise = $this->internalDeferred->promise();
        if(is_null($this->bootstrap) || $this->bootstrapped) {
            $this->internalDeferred->resolve(true);
        }
        $this->setWork($work);
        $promise->then(function() {
            $data = $this->work->getWork();
            $this->sendPacket($data);
        });
        
    }
    public function doneWork($output) {
        $this->ready=true;
        $this->running=false;
        $this->done=false;
        if(!is_null($this->queue)) {
            $this->queue->finishedWork($this->work);
        }
        $workResult = new WorkResult($output);
        $this->work->done($workResult);
    }

    private function processPacket($framedPacket) {
        $frame = WorkerFrame::fromPacket($framedPacket);
        $this->packetDebug("received",$framedPacket,$frame->get());
        $this->processOutput($frame->get());
    }

    private function processOutput($output) {
        if(!is_null($this->bootstrap) && !$this->bootstrapped) {
            if(is_array($output) && isset($output["success"]) && isset($output["data"]["bootstrap"]) && $output["success"]) {
                $this->bootstrapped = true;
                if(!is_null($this->internalDeferred)) {
                    $this->internalDeferred->resolve(true);
                }
                return;
            }
        }
        $this->doneWork($output);
    }

    /**
     * Get the value of work
     *
     * @return  mixed
     */
    public function getWork() {
        return $this->work;
    }

    /**
     * Set the value of work
     *
     * @param   mixed  $work  
     * @return  self
     */
    public function setWork($work) {
        $this->work = $work;
        return $this;
    }

    
    /**
     * Get the value of id
     *
     * @return  mixed
     */
    public function getId() {
        return $this->id;
    }

    /**
     * Set the value of id
     *
     * @param   mixed  $id  
     * @return  self
     */
    public function setId($id) {
        $this->id = $id;
        return $this;
    }


    private function getProcessMeta() {
        return [
            "onStart"=>function(Process $process) {
                dhGlobal::trace("[worker]",$this->getId(),"started");
                $data = $process->Buffer()->read(true,false);
                $this->onStart();
            },
            "onDone"=>function(Process $process,$exitCode=null,$termSignal=null) {
                dhGlobal::trace("[worker]",$this->getId(),"done");
                $this->onDone($process,$exitCode,$termSignal);
            },
            "onData"=>function(Process $process,$chunk=null) {
                $this->onData($process,$chunk);
            },
        ];
        
    }
    private function packetDebug($direction,$packet,$data) {
        if($this->debugPackets) {
            dhGlobal::outLine("\n[worker-packet]",$direction,dhGlobal::padLeft("",15,"*"),"\npacket:",$packet,"\ndata::",json_encode($data),"\n[worker-packet]",$direction,dhGlobal::padLeft("",15,"*")."\n ");
        }
    }
}