<?php
namespace boru\dhutils\queue;

use boru\dhutils\dhGlobal;
use boru\dhutils\queue\Visualizer;
use boru\dhutils\queue\QueueableInterface;
use React\EventLoop\Loop;

class Queue {
    private $max = 1;
    private $delay = 0.1;

    /** @var QueueableInterface[] */
    private $queue = [];
    /** @var QueueableInterface[] */
    private $running = [];
    /** @var QueueableInterface[] */
    private $finished = [];

    /** @var Visualizer */
    private $visualizer;
    private $startTime;

    private $numTicks=0;
    private $limitTicks=0;

    public function __construct($max=1,$delay=0.1,$visualizer=null) {
        $this->max = $max;
        $this->delay = $delay;
        if(!is_null($visualizer) && is_object($visualizer)) {
            $this->visualizer = $visualizer;
        }
    }

    /**
     * 
     * @param QueueableInterface $item 
     * @return QueueableInterface 
     */
    public function queue(QueueableInterface $item) {
        $this->queue[] = $item;
        return $item;
    }

    public function run($ticks=0) {
        $this->numTicks = 0;
        $this->limitTicks = $ticks;
        Loop::addPeriodicTimer($this->delay, function($timer) {
            $this->loopHandler();
            if(!$this->hasRunning() && !$this->hasQueued()) {
                Loop::cancelTimer($timer);
            }
            if(!$this->tickCounter()) {
                Loop::cancelTimer($timer);
                Loop::stop();
                dhGlobal::outLine("Stopped after",$this->numTicks);
            }
        });
    }
    public function runOne() {
        return $this->startOne();
    }
    public function tickCounter() {
        $this->numTicks++;
        //dhGlobal::outLine("tick",$this->numTicks,$this->limitTicks);
        if($this->limitTicks>0 && $this->numTicks>=$this->limitTicks) {
            return false;
        }
        return true;
    }

    /**
     * Return the next item in the queue, if available, otherwise false
     * 
     * @return QueueableInterface|false 
     */
    public function next() {
        return $this->loopHandler();
    }
    public function setFinished(QueueableInterface $item) {
        if(isset($this->running[$item->getId()])) {
            unset($this->running[$item->getId()]);
        }
        $this->visualizerFinished($item->getId());
        $this->finished[$item->getId()] = $item;
        $this->visualizerDisplay();
    }

    public function loopHandler($pull=false) {
        if(!$pull) {
            $this->checkFinished();
        }
        $item = null;
        if($this->hasQueued() && $this->hasRoom()) {
            $item = $this->startOne();
        }
        $this->visualizerDisplay();
        return is_null($item) ? false : $item;
    }

    public function hasQueued() {
        return count($this->queue)>0 ? true : false;
    }
    public function hasRunning() {
        return count($this->running)>0 ? true : false;
    }
    public function hasRoom() {
        if($this->max <= 0) {
            return true;
        }
        return count($this->running)<$this->max ? true : false;
    }
    public function getRunning() {
        return $this->running;
    }
    public function getQueued() {
        return $this->queued;
    }
    public function getDone() {
        return $this->finished;
    }

    private function startOne() {
        if(!empty($this->queue) && $this->hasRoom()) {
            if(is_null($this->startTime)) {
                $this->startTime = microtime(true);
            }
            $item = array_shift($this->queue);
            $this->running[$item->getId()] = $item;
            $item->start();
            $this->visualizerStarted($item->getId());
            return $item;
        }
        return false;
    }
    private function checkFinished() {
        if(!empty($this->running)) {
            foreach($this->running as $id=>$item) {
                if($item->isDone()) {
                    $this->visualizerFinished($id);
                    $this->finished[$id] = $item;
                    unset($this->running[$id]);
                }
            }
        }
    }

    public function visualizerDisplay() {
        if(!is_null($this->visualizer)) {
            $this->visualizer->display();
        }
    }
    private function visualizerStarted($id) {
        if(!is_null($this->visualizer)) {
            $this->visualizer->started($id);
        }
    }
    private function visualizerFinished($id) {
        if(!is_null($this->visualizer)) {
            $this->visualizer->finished($id);
        }
    }
    
    public function max($max=null) {
        return is_null($max) ? $this->getMax() : $this->setMax($max);
    }
    public function delay($delay=null) {
        return is_null($delay) ? $this->getDelay() : $this->setDelay($delay);
    }
    public function startTime($startTime=null) {
        return is_null($startTime) ? $this->getStartTime() : $this->setStartTime($startTime);
    }
    public function visualizer($visualizer=null) {
        return is_null($visualizer) ? $this->getVisualizer() : $this->setVisualizer($visualizer);
    }

    /**
     * Get the value of max
     *
     * @return  mixed
     */
    public function getMax() {
        return $this->max;
    }

    /**
     * Set the value of max
     *
     * @param   mixed  $max  
     * @return  self
     */
    public function setMax($max) {
        $this->max = $max;
        return $this;
    }

    /**
     * Get the value of delay
     *
     * @return  mixed
     */
    public function getDelay() {
        return $this->delay;
    }

    /**
     * Set the value of delay
     *
     * @param   mixed  $delay  
     * @return  self
     */
    public function setDelay($delay) {
        $this->delay = $delay;
        return $this;
    }

    /**
     * Get the value of visualizer
     *
     * @return  mixed
     */
    public function getVisualizer() {
        return $this->visualizer;
    }

    /**
     * Set the value of visualizer
     *
     * @param   mixed  $visualizer  
     * @return  self
     */
    public function setVisualizer($visualizer) {
        $this->visualizer = $visualizer;
        return $this;
    }

    /**
     * Get the value of startTime
     *
     * @return  mixed
     */
    public function getStartTime() {
        return $this->startTime;
    }

    /**
     * Set the value of startTime
     *
     * @param   mixed  $startTime  
     * @return  self
     */
    public function setStartTime($startTime) {
        $this->startTime = $startTime;
        return $this;
    }
}