<?php
namespace boru\dhutils\queue;

use boru\dhutils\dhGlobal;

class Visualizer {
    private $max = 1;
    private $delay = 0.5;
    private $expected = 0;

    private $startTime;
    private $finished=0;

    private $enabled = true;
    private $lastTime;
    private $data = [];
    private $itemMap = [];

    public function __construct($max,$delay=null,$expected=null) {
        $this->setMax($max);

        if(!is_null($delay)) {
            $this->setDelay($delay);
        }
        if(!is_null($expected)) {
            $this->setExpected($expected);
        }

        $this->data = array_fill(0,$max,' ');
    }

    public function max($max=null) {
        return is_null($max) ? $this->getmax() : $this->setmax($max);
    }
    public function started($itemId) {
        return $this->startItem($itemId);
    }
    public function finished($itemId) {
        return $this->endItem($itemId);
    }
    public function display() {
        if(!$this->canDisplay()) {
            return;
        }
        $this->lastTime = microtime(true);
        dhGlobal::outLine(implode(" ",$this->data),"    | ",implode(" | ",$this->stats()));
    }
    private function stats() {
        //Runtime
        $stats = [];
        $runTime = $this->formatTime($this->runTime());        

        //Percent complete and ETA
        if($this->expected > 0) {
            $percent = (int) $this->finished / (int) $this->expected;
            $percentRemaining = 1 - $percent;
            $percent *= 100;
            $percent = dhGlobal::pad(sprintf("%01.2f",$percent),6," ",STR_PAD_LEFT);
            $eta = $this->formatTime($this->runTime() * $percentRemaining);
        } else {
            $percent = "% N/A";
            $eta = "N/A";
        }

        $stats[] = "runTime: ".dhGlobal::pad($runTime,6," ",STR_PAD_LEFT);
        $stats[] = "%".dhGlobal::pad($percent,8," ",STR_PAD_LEFT);
        $stats[] = "eta: ".dhGlobal::pad($eta,6," ",STR_PAD_LEFT);
        return $stats;
    }

    private function startItem($itemId) {
        if(is_null($this->startTime)) {
            $this->startTime = microtime(true);
        }
        $index = $this->nextIndex();
        $this->itemMap[$itemId] = $index;
        $this->data[$index] = '.';
    }
    private function endItem($itemId) {
        $index = $this->itemMap[$itemId];
        $this->data[$index] = 'x';
        $this->finished++;
    }

    private function nextIndex() {
        foreach($this->data as $i=>$s) {
            if($s == ' ') {
                return $i;
            }
        }
        foreach($this->data as $i=>$s) {
            if($s == 'x') {
                return $i;
            }
        }
    }

    /**
     * time elapsed since started
     * @return float 
     */
    private function runTime() {
        return !is_null($this->startTime) ? microtime(true) - $this->startTime : 0;
    }

    private function canDisplay() {
        if(!$this->enabled) {
            return false;
        }
        if(is_null($this->lastTime)) {
            $this->lastTime = 0;
        }
        if(microtime(true) - $this->lastTime >= $this->delay) {
            return true;
        }
        return false;
    }
    private function formatTime($time) {
        if($time<=60) {
            $str = sprintf("%01.2f",$time)."s";
        } else {
            $time = round($time);
            $dtF = new \DateTime('@0');
            $dtT = new \DateTime("@$time");
            
            $str = dhGlobal::dateIntervalToElapsed($dtF->diff($dtT),true,false);
        }
        return $str;
    }

    public function delay($delay=null) {
        return is_null($delay) ? $this->getDelay() : $this->setDelay($delay);
    }
    public function expected($expected=null) {
        return is_null($expected) ? $this->getExpected() : $this->setExpected($expected);
    }
    public function enabled($enabled=null) {
        return is_null($enabled) ? $this->getEnabled() : $this->setEnabled($enabled);
    }

    /**
     * Get the value of delay
     *
     * @return  mixed
     */
    public function getDelay() {
        return $this->delay;
    }

    /**
     * Set the value of delay
     *
     * @param   mixed  $delay  
     * @return  self
     */
    public function setDelay($delay) {
        $this->delay = $delay;
        return $this;
    }

    /**
     * Get the value of expected
     *
     * @return  mixed
     */
    public function getExpected() {
        return $this->expected;
    }

    /**
     * Set the value of expected
     *
     * @param   mixed  $expected  
     * @return  self
     */
    public function setExpected($expected) {
        $this->expected = $expected;
        return $this;
    }

    /**
     * Get the value of enabled
     *
     * @return  mixed
     */
    public function getEnabled() {
        return $this->enabled;
    }

    /**
     * Set the value of enabled
     *
     * @param   mixed  $enabled  
     * @return  self
     */
    public function setEnabled($enabled) {
        $this->enabled = $enabled;
        return $this;
    }

    /**
     * Get the value of max
     *
     * @return  mixed
     */
    public function getMax() {
        return $this->max;
    }

    /**
     * Set the value of max
     *
     * @param   mixed  $max  
     * @return  self
     */
    public function setMax($max) {
        $this->max = $max;
        return $this;
    }
}