<?php
namespace boru\dhutils\tools;

use boru\dhutils\dhGlobal;
use boru\dhutils\progress\ProgressTracker;
use React\EventLoop\Loop;

class StatusBar {

    private $active = false;
    private $started = false;
    private $updateInterval=0.5;
    private $scrollAreaCreated=false;
    private $lines;
    private $cols;
    private $barContent;
    private $progressBarLines = 0;
    private $clearLinesCache;

    private $showExec = false;
    private $showMem = false;

    private $timer;

    /** @var \boru\dhutils\progress\Bar[] */
    private $bars = [];

    public function __construct($updateInterval=0.5) {
        $this->lines = $this->getLines();
        $this->cols  = $this->getCols();
        $this->updateInterval = $updateInterval;
    }
    public function __destruct() {
        $this->destroyScrollArea();
        if($this->scrollAreaCreated) {
            $this->destroyScrollArea();
        }
    }

    public function addBar($name,$bar) {
        $this->bars[$name] = $bar;
        $this->progressBarLines = count($this->bars);
        if($this->started && $this->active) {
            $this->clearLinesCache=null;
            $this->setupScrollArea(true);
        }
    }
    public function removeBar($name) {
        if(isset($this->bars[$name])) {
            unset($this->bars[$name]);
        }
        $this->progressBarLines = count($this->bars);
        if($this->started && $this->active) {
            $this->clearLinesCache=null;
            $this->setupScrollArea(true);
        }
    }

    public function start() {
        if($this->progressBarLines<=0) {
            return;
        }
        $this->started=true;
        $this->active=true;
        $this->setupScrollArea();
        $this->draw();
        Loop::addPeriodicTimer($this->updateInterval,function($timer) {
            if($this->active) {
                $this->draw();
                if(is_null($this->timer)) {
                    $this->timer = $timer;
                }
            } else {
                Loop::cancelTimer($timer);
                $this->stop();
            }
        });
    }
    public function stop() {
        $this->started = false;
        $this->active = false;
        //StdOut::line($this->getBarContent());
        if(!is_null($this->timer) && $this->timer instanceof \React\EventLoop\TimerInterface) {
            Loop::cancelTimer($this->timer);
        }
        $this->timer = null;
        if($this->scrollAreaCreated) {
            $this->destroyScrollArea();
        }
        
    }

    public function draw() {
        if(!$this->active) {
            return;
        }
        if($this->progressBarLines<=0) {
            $this->stop();
            return;
        }
        if(!$this->started) {
            $this->setupScrollArea();
            $this->started=true;
        }

        $this->checkResetScrollArea();
        $this->drawStatusBars();
    }

    private function drawStatusBars($content=null) {
        if(is_null($content)) {
            $content = $this->getBarContent();
        }
        $this->printStatusBar($content);
    }

    private function printStatusBar($content=null) {
        //echo StdOut::CODE_SAVE_CURSOR;
        echo "\033[s";
        echo $this->clearLines();
        if(!is_null($content)) {
            echo $content;
        }
        //echo StdOut::CODE_RESTORE_CURSOR;
        echo "\033[u";
    }

    public function getBarContent() {
        $start = microtime(true);
        $content = [];
        foreach($this->bars as $bar) {
            $content[] = $bar->content();
        }
        $this->getCols(5);
        $this->barContent = implode("\n",$content);
        $endContent = "";
        if($this->showExec) {
            $end = microtime(true);
            $exec = $end-$start;
            $endContent.=" [exec: ".number_format($exec,5)."]";
        }
        if($this->showMem) {
            $mem = memory_get_usage();
            $endContent.=" [mem: ".dhGlobal::formatBytes($mem,"%01.2f%s")."]";
        }
        $maxLen = $this->getCols(strlen($endContent)+3);
        return substr($this->barContent,0,$maxLen) . $endContent;
    }

    private function checkResetScrollArea() {
        $curLines = $this->getLines();
        $newLines = $this->getLines(0,true);
        if($curLines != $newLines) {
            $this->setupScrollArea(true);
            $this->clearLinesCache=null;
        }
    }

    private function clearLines() {
        if(is_null($this->clearLinesCache)) {
            $this->clearLinesCache = "";
            if($this->progressBarLines>0) {
                $i = 0;
                while($i<$this->progressBarLines) {
                    $lineNum = $this->getLines($i);
                    $this->clearLinesCache.="\033[".$lineNum.";0f";
                    $this->clearLinesCache.=StdOut::ERASE_LINE_AFTER;
                    $i++;
                }
            }
        }
        return $this->clearLinesCache;
    }

    private function setupScrollArea($force=false) {
        if($this->scrollAreaCreated && !$force) {
            return;
        }
        echo str_repeat("\n",$this->progressBarLines);
        //save cursor position
        echo StdOut::CODE_SAVE_CURSOR;
        echo "\033[s";

        //set scroll region
        echo "\033[0;".$this->getLines($this->progressBarLines,true)."r";
        echo StdOut::CODE_RESTORE_CURSOR;
        //move to scroll area
        echo str_repeat(StdOut::MOVE_UP,$this->progressBarLines);
        $this->scrollAreaCreated=true;
    }
    private function destroyScrollArea($force=false) {
        if(!$this->scrollAreaCreated && !$force) {
            return;
        }
        echo $this->clearLines();
        echo StdOut::CODE_SAVE_CURSOR;
        echo "\033[0;".$this->getLines(0,true)."r";
        echo StdOut::CODE_RESTORE_CURSOR;
        //move to scroll area
        echo str_repeat(StdOut::MOVE_UP,$this->progressBarLines);
        //$this->printStatusBar();
        echo StdOut::ERASE_LINE_AFTER.$this->getBarContent();
        $this->scrollAreaCreated=false;
        echo str_repeat("\n",$this->progressBarLines);
        //echo $this->getBarContent()."\n";
    }

    private function getLines($offset=0,$refresh=false) {
        if(is_null($this->lines) || $refresh) {
            $this->lines = exec("tput lines 2>/dev/null");
        }
        if(is_null($this->lines) || !is_numeric($this->lines)) {
            $this->lines = 24;
        }
        return intval($this->lines-$offset);
    }
    private function getCols($offset=0,$refresh=false) {
        if(is_null($this->cols) || $refresh) {
            $this->cols = exec("tput cols 2>/dev/null");
        }
        if(is_null($this->cols) || !is_numeric($this->cols)) {
            $this->cols = 24;
        }
        return intval($this->cols-$offset);
    }

    /**
     * @param float $updateInterval 
     * @return $this 
     */
    public function updateInterval($updateInterval) {
        $this->updateInterval=$updateInterval;
        return $this;
    }

    /**
     * @param bool $showExec 
     * @return $this 
     */
    public function showExec($showExec) {
        $this->showExec=$showExec?true:false;
        return $this;
    }
    /**
     * @param bool $showExec 
     * @return $this 
     */
    public function showMem($showMem) {
        $this->showMem=$showMem?true:false;
        return $this;
    }
}