<?php
namespace boru\dhutils\tools;

use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\parts\WorkResult;
use boru\dhutils\queue\Queue;
use boru\dhutils\queue\Visualizer;
use boru\dhutils\multithread\process\Work;
use boru\dhutils\multithread\process\Worker;
use React\EventLoop\Loop;
use React\Promise\Deferred;

class WorkQueue {
    /** @var Queue */
    private $queue;

    /** @var Worker[] */
    private $processes;
    /** @var bool */
    private $started = false;
    /** @var bool */
    private $running = false;

    private $max = 1;
    private $delay = 0.1;
    private $expected = 0;
    private $visualize = false;
    private $visualizerDelay = 1.0;
    private $collect = true;
    private $debugPackets = false;

    /** @var WorkResult[] */
    private $collected = [];

    /** @var \React\Promise\Deferred */
    private $deferred;

    /**
     * Variables for ClassProcess creation
     * @var mixed
     */
    private $bootstrap;


    public function __construct($max=1,$options=[],$queue=null,$visualizer=null) {
        $this->max = $max;
        $this->delay            = dhGlobal::getVal($options,"delay",$this->delay);
        $this->expected         = dhGlobal::getVal($options,"expected",$this->expected);
        $this->visualize        = dhGlobal::getVal($options,"visualize",false);
        $this->visualizerDelay  = dhGlobal::getVal($options,"visualizerDelay",$this->visualizerDelay);
        $this->bootstrap        = dhGlobal::getVal($options,"bootstrap",null);
        $this->collect          = dhGlobal::getVal($options,"collect",$this->collect);
        $this->debugPackets     = dhGlobal::getVal($options,"debugPackets",$this->debugPackets);

        $this->initQueue($queue,$visualizer);
        $this->initProcesses();
    }

    public function initQueue($queue=null,$visualizer=null) {
        if(is_null($queue)) {
            $this->queue = new Queue($this->max,$this->delay);
        } else {
            $this->queue = $queue;
        }
        if(method_exists($this->queue,"setVisualizer")) {
            if(is_null($visualizer)) {
                if($this->visualize) {
                    $visualizer = new Visualizer($this->max,$this->visualizerDelay,$this->expected);
                    $this->queue->setVisualizer($visualizer);
                }
            } else {
                $this->queue->setVisualizer($visualizer);
            }
        }
    }

    public function initProcesses() {
        for($i=0;$i<$this->max;$i++) {
            $this->processes[$i] = new Worker($this->bootstrap,$this,$this->debugPackets);
            $this->processes[$i]->start();
        }
        $this->started=true;
    }
    public function terminateProcesses($graceful=true) {
        foreach($this->processes as $i=>$worker) {
            if(!$graceful) {
                $worker->stop();
            } else {
                while($worker->isRunning()) {
                    dhGlobal::trace("gracefully waiting for worker ",$i);
                    usleep($this->delay*1000000);
                }
                $worker->stop();
            }
            unset($this->processes[$i]);
        }
        $this->started=false;
    }

    /**
     * 
     * @param Work $work 
     * @return Work 
     */
    public function queue(Work $work) {
        if($this->collect) {
            $work->then(function($result) use ($work) {
                $this->collected[$work->getId()] = $result;
            });
        }
        $this->queue->queue($work);
        return $work;
    }
    public function wait() {
        if(!$this->running) {
            $this->start();
        }
        \React\Async\await($this->deferred->promise());
    }
    public function collect() {
        $this->wait();
        $collected = $this->collected;
        $this->collected=[];
        return $this->collect ? $collected : false;
    }
    public function start() {
        $this->deferred = new Deferred();
        $this->running=true;
        if(!$this->isStarted()) {
            $this->initProcesses();
        }
        Loop::addPeriodicTimer($this->delay, function($timer) {
            $this->needsWork();
            $this->queue->visualizerDisplay();
            if(!$this->queue->hasRunning() && !$this->queue->hasQueued()) {
                dhGlobal::trace("[queue]","terminate time..");
                $this->stop(true);
                Loop::cancelTimer($timer);
                dhGlobal::trace("[queue]","terminated..");
                $this->running=false;
                $this->deferred->resolve(true);
            }
        });
    }
    public function stop($graceful=true) {
        $this->terminateProcesses($graceful);
        $this->running=false;
    }
    public function finishedWork(Work $work) {
        $this->queue->setFinished($work);
    }
    public function needsWork() {
        if($this->queue->hasQueued()) {
            //dhGlobal::trace("[queue]","haveQueued..");
            foreach($this->processes as $pid=>$worker) {
                if($worker->isReady()) {
                    dhGlobal::trace("[queue]","starting work.");
                    if(($work = $this->queue->next()) !== false) {
                        $worker->startWork($work);
                    }
                }
            }
        } else {
            //dhGlobal::trace("[queue]","noQueue..");
            return false;
        }
        return true;
    }

    public function isStarted() {
        return $this->started;
    }

    /**
     * Create work and then queue it.. chainable with ->then();
     * @param mixed $callable 
     * @param array $args 
     * @param bool $asJson 
     * @return Work 
     */
    public function queueWork($callable,$args=[],$asJson=true) {
        $work = static::work($callable,$args,$asJson);
        return $this->queue($work);
    }

    /**
     * Create Work
     * @param mixed $callable 
     * @param array $args 
     * @param bool $asJson 
     * @return Work 
     */
    public static function work($callable,$args=[],$asJson=true) {
        $work = new Work($callable,["args"=>$args,"asJson"=>$asJson]);
        return $work;
    }
}