<?php
namespace boru\dhutils\traits;

use boru\dhutils\async\Queue;
use boru\dhutils\async\Work;
use \boru\dhutils\dhGlobal;
use boru\dhutils\http\Request;
use UnexpectedValueException;
use RuntimeException;
use Exception;

trait AsyncUtils {
    /**
     * Create's async Work [static::exec], and optionally runs it (if $run=true)
     * @param mixed $command 
     * @param bool $run default=false, if true will auto-queue/run it.. otherwise returns Work for manual queue or execution
     * @return Work 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     * @throws Exception 
     */
    public static function asyncExec($command,$run=false) {
        $work = new Work(["static","exec"],["args"=>[$command]]);
        if($run) {
            return static::asyncWork($work);
        } else {
            return $work;
        }
    }

    /**
     * Create's async Work [static::http], and optionally runs it (if $run=true)
     * @param Request $request 
     * @param bool $bodyAsJson default=false, if true will parse the http result body as json, returning an associative array
     * @param bool $run default=false, if true will auto-queue/run it.. otherwise returns Work for manual queue or execution
     * @return Work 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     * @throws Exception 
     */
    public static function asyncHttp(\boru\dhutils\http\Request $request,$bodyAsJson=false,$run=false) {
        $work = new Work(["static","http"],["args"=>[[
            "method"=>$request->getMethod(),
            "url"=>$request->getUrl(),
            "requestData"=>$request->data,
            "getJson"=>$bodyAsJson
        ]]]);
        if($run) {
            return static::asyncWork($work);
        } else {
            return $work;
        }
    }

    public static function asyncVisualize($visualize) {
        static::set("asyncQueue.visualize",$visualize);
    }
    public static function asyncMax($max) {
        static::set("asyncQueue.maxPerQueue",$max);
    }
    public static function asyncRetries($retries) {
        static::set("asyncQueue.retriesOnError",$retries);
    }
    public static function asyncBootstrap($bootstrap) {
        static::set("asyncQueue.defaultBootstrap",$bootstrap);
    }
    public static function asyncSetSettings($maxPerQueue=3,$retriesOnError=0,$defaultBootstrap=null) {
        static::set("asyncQueue.maxPerQueue",$maxPerQueue);
        static::set("asyncqueue.retriesOnError",$retriesOnError);
        static::set("asyncqueue.defaultBootstrap",$defaultBootstrap);
    }
    /**
     * Shortcut to queue Work
     * @param Work $work 
     * @param mixed $bootstrap  
     * @return Work 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     * @throws Exception 
     */
    public static function asyncWork(Work $work,$bootstrap=null) {
        $queue = static::getAsyncQueue($bootstrap);
        $queue->queue($work);
        return $work;
    }
    /**
     * Shortcut to create and queue Work
     * @param mixed $callable 
     * @param array $args 
     * @param mixed $bootstrap 
     * @return Work 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     * @throws Exception 
     */
    public static function asyncWorkCallable($callable,$args=[],$bootstrap=null) {
        $work = new Work($callable,["args"=>$args]);
        return static::asyncWork($work);
    }
    /**
     * Get a Queue (or create one if not found) based on $bootstrap
     * @param mixed $bootstrap 
     * @return Queue
     */
    public static function getAsyncQueue($bootstrap=null) {
        if(is_array($bootstrap)) {
            ksort($bootstrap);
            $queueName = sha1(json_encode($bootstrap));
        } elseif(!is_null($bootstrap)) {
            $queueName = sha1($bootstrap);
        } else {
            $queueName = "default";
            $bootstrap = static::get("asyncqueue.defaultBootstrap",null);
        }
        if(($queue = static::get("asyncQueue.".$queueName,false)) === false) {
            $queue = new Queue(static::get("asyncQueue.maxPerQueue",3),$bootstrap,static::get("asyncqueue.retriesOnError",0));
            static::set("asyncQueue.".$queueName,$queue);
        }
        return $queue;
    }

    /**
     * 
     * @param Work[]|Work $workArray 
     * @return void 
     */
    public static function waitForWork($workArrayOrInstance) {
        if(!is_array($workArrayOrInstance)) {
            $workArray = [$workArrayOrInstance];
        } else {
            $workArray = $workArrayOrInstance;
        }
        $promises=[];
        foreach($workArray as $work) {
            $promises[] = $work->promise();
        }
        return static::waitForAsync($promises);
    }
    /**
     * 
     * @param Work[]|Work $workArray 
     * @return void 
     */
    public static function collectWork($workArrayOrInstance) {
        return static::waitForWork($workArrayOrInstance);
    }
    public static function waitForAsync($promises=[]) {
        $results = [];
        foreach($promises as $promise) {
            $results[] = \React\Async\await($promise);
        }
        return $results;
    }
    public static function collectAsync($promises=[]) {
        return static::waitForAsync();
    }
    public static function waitForQueue($bootstrap=null) {
        $queue = static::getAsyncQueue($bootstrap);
        $workArray = $queue->getAllWork();
        return static::waitForWork($workArray);
    }
    public static function collectQueue($bootstrap=null) {
        return static::waitForQueue($bootstrap);
    }
}