<?php
namespace boru\dhutils\traits;

use boru\dhutils\filesys\File;
use boru\dhutils\filesys\Directory;

trait GlobalFilesys {
    
    /**
     * Checkes to see if the provided Path exists as a filename.
     * @param mixed $pathString 
     * @param bool $readMeta 
     * @return false|File - Note, will return false if the path is a Directory.
     */
    public static function fileIfExists($pathString,$readMeta=true) {
        if(!file_exists($pathString) || is_dir($pathString)) {
            return false;
        }
        return new \boru\dhutils\filesys\File(["path"=>$pathString,"readMeta"=>$readMeta]);
    }

    /**
     * Checkes to see if the provided Path exists as a Directory.
     * Returns a \boru\dhutils\filesys\Directory object if is a directory and exists, otherwise false
     * 
     * Note, will return false if the path is a File or Symlink.
     */
    /**
     * Checkes to see if the provided Path exists as a Directory.
     * @param mixed $pathString 
     * @param bool $scan 
     * @return false|Directory Note, will return false if the path is a File or Symlink.
     */
    public static function dirIfExists($pathString,$scan=true) {
        if(!file_exists($pathString) || !is_dir($pathString)) {
            return false;
        }
        return new \boru\dhutils\filesys\Directory(["path"=>$pathString,"scan"=>$scan]);
    }

    public static function getFileHashAndSize($data, $offset = 0, $partSize = null) {
        if (!$partSize) {
            if (is_resource($data)) {
                // We need to calculate the file's hash incrementally from the stream.
                $context = hash_init('sha1');
                hash_update_stream($context, $data);
                $hash = hash_final($context);
                // Similarly, we have to use fstat to get the size of the stream.
                $size = fstat($data)['size'];
                // Rewind the stream before passing it to the HTTP client.
                rewind($data);
            } else {
                // We've been given a simple string body, it's super simple to calculate the hash and size.
                $hash = sha1($data);
                $size = mb_strlen($data, '8bit');
            }
        } else {
            $dataPart = static::getPartOfFile($data, (int) $offset, (int) $partSize);
            $hash = sha1($dataPart);
            $size = mb_strlen($dataPart, '8bit');
        }

        return array("hash"=>$hash, "size"=>$size);
    }

    /**
     * Return selected part of file
     *
     * @param $data
     * @param int $offset
     * @param int $partSize
     * @return bool|string
     */
    public static function getPartOfFile($data, $offset, $partSize) {
        // Get size and hash of one data chunk
        if (is_resource($data)) {
            // Get data chunk
            fseek($data, $offset);
            $dataPart = fread($data, $partSize);
            // Rewind the stream before passing it to the HTTP client.
            rewind($data);
        } else {
            $dataPart = substr($data, $offset, $partSize);
        }
        return $dataPart;
    }

    /**
     * Return a santized relative path
     * 
     * Used to make sure the path is within the defined root.
     * 
     * @param string $path The path to test
     * @param string $root default=current directory
     * @param boolean $relative return relative path to $path if true, otherwise full path
     * @return mixed sanitized path if within root, false if not
     */
    public static function sanitizePath($path,$root=null,$relative=true) {
        $bp = new \boru\dhutils\filesys\BasePath($path);
        if(is_null($root)) {
            $root = ".";
        }
        $rpath = $bp->relative($root);
        if($rpath === false) {
            return false;
        }
        return $relative ? $rpath : $bp->fullPath();
    }
}