<?php
namespace boru\dhutils\traits;

use boru\dhutils\dhGlobal;

trait GlobalUtils {
    /**
     * Determine if an array is an assoc array (true) or not (false).
     * 
     * Example:
     * 
     * dhGlobal::isAssoc(['test'=>1]) -- returns true
     * 
     * @param array $array the array to test
     * @return boolean true if is assoc, false if not
     */
    public static function isAssoc($array) {
		if(!is_array($array)) return false;
		return (bool)count(array_filter(array_keys($array), 'is_string'));
	}

    /**
     * Determine if an array is a numeric list array (true) or an assoc array (false).
     * 
     * Example:
     * 
     * dhGlobal::isAssoc(['test'=>1]) -- returns false
     * 
     * @param array $array the array to test
     * @return boolean true if is assoc, false if not
     */
    public static function isList($arr) {
        if ($arr === []) {
            return true;
        }
        return array_keys($arr) === range(0, count($arr) - 1);
    }
    

    

    public static function exec($cmd,$array=true) {
        if(!$array) {
            ob_start();
            passthru($cmd);
            $data = ob_get_contents();
            ob_end_clean();
            return $data;
        } else {
            $data = [];
            exec($cmd,$data);
            return $data;
        }
    }

    /**
     * Truncate a string to a certain length if necessary,
     * optionally splitting in the middle of a word, and
     * appending the $etc string or inserting $etc into the middle.
     * 
     * @param string  $string      input string
     * @param integer $length      length of truncated text
     * @param string  $etc         end string
     * @param boolean $break_words truncate at word boundary
     * @param boolean $middle      truncate in the middle of text
     * @return string truncated string
     */
    public static function truncate($string, $length = 80, $etc = '...', $break_words = false, $middle = false) {
        if ($length == 0)
            return '';
    
        if (function_exists('mb_strlen')) {
            if (mb_strlen($string, 'UTF-8') > $length) {
                $length -= min($length, mb_strlen($etc, 'UTF-8'));
                if (!$break_words && !$middle) {
                    $string = preg_replace('/\s+?(\S+)?$/u', '', mb_substr($string, 0, $length + 1, 'UTF-8'));
                } 
                if (!$middle) {
                    return mb_substr($string, 0, $length, 'UTF-8') . $etc;
                }
                return mb_substr($string, 0, $length / 2, 'UTF-8') . $etc . mb_substr($string, - $length / 2, $length, 'UTF-8');
            }
            return $string;
        }
        
        // no MBString fallback
        if (isset($string[$length])) {
            $length -= min($length, strlen($etc));
            if (!$break_words && !$middle) {
                $string = preg_replace('/\s+?(\S+)?$/', '', substr($string, 0, $length + 1));
            } 
            if (!$middle) {
                return substr($string, 0, $length) . $etc;
            }
            return substr($string, 0, $length / 2) . $etc . substr($string, - $length / 2);
        }
        return $string;
    } 

    /**
     * Pad a $string to $len length using $padstring
     * * Balances the $string's length to match an even pad if $padstring is more than 1 char
     */
    public static function pad($string,$len,$padstring=". ",$padStyle=STR_PAD_RIGHT) {
        if(strlen($padstring)>1) {
            $mod = strlen($string) % strlen($padstring);
            if($mod !== 0) {
                $diff = strlen($padstring)-$mod;
                $string = str_pad($string,strlen($string)+$diff," ",$padStyle);
            }
        }
        return str_pad($string,$len,$padstring,$padStyle);
    }
    /**
     * Right-Pad a $string to $len length using $padstring
     * * Balances the $string's length to match an even pad if $padstring is more than 1 char
     */
    public static function padRight($string,$len,$padstring=". ") {
        return static::pad($string,$len,$padstring,STR_PAD_RIGHT);
    }
    /**
     * Left-Pad a $string to $len length using $padstring
     * * Balances the $string's length to match an even pad if $padstring is more than 1 char
     */
    public static function padLeft($string,$len,$padstring=". ") {
        return static::pad($string,$len,$padstring,STR_PAD_LEFT);
    }
    /**
     * Middle-Pad a $string to $len length using $padstring
     * * Balances the $string's length to match an even pad if $padstring is more than 1 char
     */
    public static function padMid($string,$len,$padstring=". ") {
        return static::pad($string,$len,$padstring,STR_PAD_BOTH);
    }
    /**
     * Used to limit the number of concurrent exec() calls for a given 'tag'
     * * Output: (int) number of running instances when < $concurrent
     * * Inputs
     * * * $tag (string) - identifier that should be included in the exec() command
     * * * $concurrent (int) - maximum concurrent instances of $tag that can exist
     * * * $seconds (float) - time to sleep in seconds between checks until < $concurrent
     * * * $callback (lambda) - callback function to run on each loop after checking
     */
    public static function throttle($tag,$concurrent=5,$seconds=1,$callback=null) {
        $check_command = 'ps aux | grep "'.$tag.'" | grep -v grep | wc -l';
        $data = exec($check_command);
        if(!is_null($callback)) {
            $callback();
        }
        while($data>=$concurrent && $data>0) {
            usleep($seconds*1000000);
            $data = exec($check_command);
            if(!is_null($callback)) {
                $callback();
            }
        }
        return $data;
    }

    

    /**
     * Trim a specific string off the beginning, end or both.
     * 
     * @param string $needle the string to trim off
     * @param string $haystack the string to trim $needle from
     * @param int $flag (default=both) dhGlobal::TRIM_BOTH, dhGlobal::TRIM_START, dhGlobal::TRIM_END
     * @param int $limit (default=0) Number of times to trim (0=unlimited)
     * @return string
     */
    public static function trimString($needle,$haystack,$flag=dhGlobal::TRIM_BOTH,$limit=0) {
        $needleLen = strlen($needle);
        if($flag == dhGlobal::TRIM_BOTH || $flag == dhGlobal::TRIM_START) {
            $cycles=0;
            while(strlen($haystack)>=$needleLen && substr($haystack,0,$needleLen) == $needle) {
                $haystack = substr($haystack,$needleLen);
                $cycles++;
                if($limit>0 && $cycles>=$limit) {
                    break;
                }
            }
        }
        if($flag == dhGlobal::TRIM_BOTH || $flag == dhGlobal::TRIM_END) {
            $cycles=0;
            while(strlen($haystack)>=$needleLen && substr($haystack,-$needleLen) == $needle) {
                $haystack = substr($haystack,0,-$needleLen);
                $cycles++;
                if($limit>0 && $cycles>=$limit) {
                    break;
                }
            }
        }
        return $haystack;
    }

    /**
     * Trim a specific string ($needle) off the beginning and end of $haystack, wrapper for dhGlobal::trimString()
     * 
     * @param string $needle the string to trim off
     * @param string $haystack the string to trim $needle from
     * @param int $limit (default=0) Number of times to trim (0=unlimited) 
     * @return string 
     */
    public static function btrimString($needle,$haystack,$limit=0) {
        return static::trimString($needle,$haystack,dhGlobal::TRIM_BOTH,$limit);
    }

    /**
     * Trim a specific string ($needle) off the beginning of $haystack, wrapper for dhGlobal::trimString()
     * 
     * @param string $needle the string to trim off
     * @param string $haystack the string to trim $needle from
     * @param int $limit (default=0) Number of times to trim (0=unlimited) 
     * @return string 
     */
    public static function ltrimString($needle,$haystack,$limit=0) {
        return static::trimString($needle,$haystack,dhGlobal::TRIM_START,$limit);
    }

    /**
     * Trim a specific string ($needle) off the end of $haystack, wrapper for dhGlobal::trimString()
     * 
     * @param string $needle the string to trim off
     * @param string $haystack the string to trim $needle from
     * @param int $limit (default=0) Number of times to trim (0=unlimited) 
     * @return string 
     */
    public static function rtrimString($needle,$haystack,$limit=0) {
        return static::trimString($needle,$haystack,dhGlobal::TRIM_END,$limit);
    }

    /**
     * Convert bytes into B/KB/MB/GB/TB
     * 
     * @param mixed $bytes bytes to convert
     * @param int $precision (2)
     * @param null|string $forceUnit 'B', 'KB', 'MB', 'GB', 'TB'
     * @param boolean $space (false) space between value and unit
     * @return string 
     */
    public static function formatBytes($bytes,$format="%01.2f %s",$forceUnit=null) { 
        if(is_null($bytes)) {
            $bytes = 0;
        } elseif(!is_numeric($bytes)) {
            $bytes = 0;
        }
        $units = array('B', 'KB', 'MB', 'GB', 'TB'); 
    
        $bytes = max($bytes, 0); 
        if(($pow = array_search((string) $forceUnit,$units)) === false) {
            $pow = floor(($bytes ? log($bytes) : 0) / log(1024)); 
            $pow = min($pow, count($units) - 1); 
        }
    
        // Uncomment one of the following alternatives
        $bytes /= pow(1024, $pow);
        // $bytes /= (1 << (10 * $pow)); 

        return sprintf($format, $bytes,$units[$pow]);
    }

    public static function camelize($input, $lowercaseFirst=true, $separator = [' ','-','_'])
    {
        if($lowercaseFirst) {
            return lcfirst(str_replace($separator, '', ucwords($input, implode('',$separator))));
        } else {
            return str_replace($separator, '', ucwords($input, implode('',$separator)));
        }
    }
    public static function array_is_list($array) {
        if(!is_array($array)) {
            return false;
        }
        if ([] === $array || $array === array_values($array)) {
            return true;
        }

        $nextKey = -1;

        foreach ($array as $k => $v) {
            if ($k !== ++$nextKey) {
                return false;
            }
        }

        return true;
    }
}