<?php
namespace boru\dot;

class Dot {
    /**
     * Flatten a multi-dimensional associative array to dot notation
     * @param mixed $array The multi-dimensional array
     * @param string $prepend (mostly used for recursive processing) The string to prepend to the key
     * @return array 
     */
    public static function fromArray($array, $prepend = '') {
        $results = [];

        foreach ($array as $key => $value) {
            if (is_array($value) && ! empty($value)) {
                $results = array_merge($results, static::fromArray($value, $prepend.$key.'.'));
            } else {
                $results[$prepend.$key] = $value;
            }
        }

        return $results;
    }

    /**
     * Parse a dot notation array to a multi-dimensional associative array
     * @param array $array The dot notation array
     * @param string $separator The separator used in the dot notation
     * @return void
     */
    public static function parse(&$array,$separator='.') {
		foreach($array as $k=>$v) {
			if(is_array($v)) {
				$array[$k] = static::parse($v,$separator);
			}
			if(strpos($k,$separator) !== false) {
				static::set($array,$k,$v,$separator);
				unset($array[$k]);
			}
		}
	}

    /**
     * Explodes dot notation and assigns a value based on the key
     * 
     * Example:
     * 
     * $arr = ['key'=>['subkey'=>'value']];
     * 
     * dhGlobal::set($arr,'key.subkey2','value2)
     * 
     * //['key'=>['subkey'=>'value','subkey2'=>'value2']]
     * 
     * @param array $arr the array to get the element from
     * @param string $path the dotNotated key
     * @param mixed $value the value to assign
     * @param string $separator defaults to '.'
     * @return void
     */
	public static function set(&$arr, $path, $value=null, $separator='.') {
		$keys = explode($separator, $path);

		foreach ($keys as $key) {
			$arr = &$arr[$key];
		}

		$arr = $value;
	}

    /**
     * Get an element from $arr based on dot notation
     * 
     * Example:
     * 
     * dhGlobal::get(['key'=>['subkey'=>'value']],'key.subkey') -- returns 'value'
     * 
     * @param array $arr the array to get the element from
     * @param string $dotString the key to find
     * @param mixed $default a default value if key isn't found
     * @return mixed the value if found, $defualt if not
     */
    public static function get($arr=[],$dotString,$default=null,$separator=".") {
		$pieces = explode($separator,$dotString);
		$pointer = $arr;
		for($i=0;$i<count($pieces);$i++) {
			if(isset($pointer[$pieces[$i]])) {
				$pointer = $pointer[$pieces[$i]];
			} else {
				return $default;
			}
		}
		return $pointer;
	}
}