<?php
namespace boru\dot;

class Dot {
    /**
     * Flatten a multi-dimensional associative array to dot notation
     * @param mixed $array The multi-dimensional array
     * @param string $prepend (mostly used for recursive processing) The string to prepend to the key
     * @return array 
     */
    public static function fromArray($array, $prepend = '') {
        $results = [];

        foreach ($array as $key => $value) {
            if (is_array($value) && ! empty($value)) {
                $results = array_merge($results, static::fromArray($value, $prepend.$key.'.'));
            } else {
                $results[$prepend.$key] = $value;
            }
        }

        return $results;
    }

    /**
     * Parse a dot notation array to a multi-dimensional associative array
     * @param array $array The dot notation array
     * @param string $separator The separator used in the dot notation
     * @return void
     */
    public static function parse(&$array,$separator='.') {
		foreach($array as $k=>$v) {
			if(is_array($v)) {
				$array[$k] = static::parse($v,$separator);
			}
			if(strpos($k,$separator) !== false) {
				static::set($array,$k,$v,$separator);
				unset($array[$k]);
			}
		}
	}

    /**
     * Explodes dot notation and assigns a value based on the key
     * 
     * Example:
     * 
     * $arr = ['key'=>['subkey'=>'value']];
     * 
     * dhGlobal::set($arr,'key.subkey2','value2)
     * 
     * //['key'=>['subkey'=>'value','subkey2'=>'value2']]
     * 
     * @param array $arr the array to get the element from
     * @param string $path the dotNotated key
     * @param mixed $value the value to assign
     * @param string $separator defaults to '.'
     * @return void
     */
	public static function set(&$arr, $path, $value=null, $separator='.') {
		$keys = explode($separator, $path);

		foreach ($keys as $key) {
			$arr = &$arr[$key];
		}

		$arr = $value;
	}

    /**
     * Get an element from $arr based on dot notation
     * 
     * Example:
     * 
     * dhGlobal::get(['key'=>['subkey'=>'value']],'key.subkey') -- returns 'value'
     * 
     * @param array $arr the array to get the element from
     * @param string $dotString the key to find
     * @param mixed $default a default value if key isn't found
     * @return mixed the value if found, $defualt if not
     */
    public static function get($arr=[],$dotString,$default=null,$separator=".") {
		$pieces = explode($separator,$dotString);
		$pointer = $arr;
		for($i=0;$i<count($pieces);$i++) {
			if(isset($pointer[$pieces[$i]])) {
				$pointer = $pointer[$pieces[$i]];
			} else {
				return $default;
			}
		}
		return $pointer;
	}

    /**
     * get multiple.. accepts an array of parameters to search
     * 
     * returns an array of all found values for matching parameters..
     * 
     * @param array $arr array to search
     * @param array $keysOrKeyArrays aray of "keys" to search for. Each key can be an array of key=>default value if a default is needed.
     * @param boolean $single (default=false) if true, returns the first matching value in order of $keyDefault key definitions
     * @param mixed $default (default=null) default value if key isn't found
     * @param string $keySeparator (default='.') string to split for 'dot search'
     * @return mixed if $single==false array of values for each of the keyDefault pairs. Otherwise the first matching value
     */
    public static function multiGet(array $arr,$keysOrKeyArrays,$single=false,$default=null,$keySeparator=".") {
        //first lets set up our $keyValueArray.. if cylce through $keysOrKeyArrays.. if it's a string, we'll set it to the key and the default to the $default
        //if it's an array, we'll set the key to the key and the default to the value
        $keyValueArray = [];
        foreach($keysOrKeyArrays as $entry=>$value) {
            //if $entry is a string and not an integer, then it's a key
            if(is_string($entry)) {
                $keyValueArray[$entry] = $value;
            } else {
                $keyValueArray[$value] = $default;
            }
        }

        $values = [];
        foreach($keyValueArray as $keyString=>$default) {
            $value = static::get($arr,$keyString,$default,$keySeparator);
            if($single && $value!==$default) {
                return $value;
            }
            $values[] = $value;
        }
        return $values;
    }


    /**
     * Delete an element from an array based on dot notation
     * 
     * Example:
     * 
     * $arr = ['key'=>['subkey'=>'value']];
     * 
     * dhGlobal::delete($arr,'key.subkey');
     * 
     * //['key'=>[]]
     * 
     * @param array $arr the array to delete the element from
     * @param string $path the dotNotated key
     * @param string $separator defaults to '.'
     * @return void
     */
    public static function delete(&$arr, $path, $separator='.') {
		$keys = explode($separator, $path);
        $end = array_pop($keys);
		foreach ($keys as $key) {
			$arr = &$arr[$key];
            
		}
        unset($arr[$end]);
	}
}