# Rendering in boru/dweb

Rendering in `boru/dweb` is **explicit, layered, and contract-driven**.  
There is no magic controller rendering, no implicit view resolution, and no
framework-owned templates.

This document explains how rendering works today and how it is designed to evolve.

---

## Core Principles

- Rendering is **not** the responsibility of the router
- Views explicitly request rendering
- Templates are module-scoped
- Layouts are explicit
- Rendering engines are swappable
- Overrides are deterministic

---

## Key Contracts

Rendering is built around two core interfaces.

### RendererInterface

Responsible for producing rendered output.

Typical responsibilities:

- Render templates
- Apply layouts
- Pass variables into templates
- Return rendered HTML as a string

### TemplateLocatorInterface

Responsible for locating template files.

Typical responsibilities:

- Resolve module template paths
- Apply override precedence
- Hide filesystem structure from views and renderers

---

## Rendering Flow

The full rendering flow looks like this:

```

View
→ RendererInterface
→ TemplateLocatorInterface
→ Template file
→ Layout template
→ HtmlResponse

```

The **View** decides:

- which template to render
- which layout to use
- what data to pass

The **Renderer** decides:

- how templates are rendered (Smarty, PHP, etc.)
- how layouts are applied

The **Locator** decides:

- where templates live
- which template wins (override vs default)

---

## Views and Rendering

Views are responsible for calling the renderer explicitly.

Example:

```php
$html = $renderer->renderLayout(
    'Skeleton',
    'layouts/default',
    'home/index',
    [
        'title' => 'Skeleton Module',
        'message' => 'Hello world'
    ]
);

return new HtmlResponse($html);
```

Nothing is inferred:

* Module name is explicit
* Layout is explicit
* Template is explicit
* Data is explicit

---

## Template Naming

Templates are referenced by **logical names**, not file paths.

Example:

```
layouts/default
home/index
```

The renderer and locator resolve these to actual files.

This allows:

* consistent overrides
* renderer swaps
* filesystem reorganization without breaking views

---

## Module-Scoped Templates

Each module owns its templates.

Example layout:

```
Modules/Skeleton/templates/
  layouts/
    default.tpl
  home/
    index.tpl
```

A module never reaches into another module’s templates.

---

## Template Overrides (Host Application)

The host application may override module templates.

Typical override structure:

```
templates/
  Skeleton/
    layouts/
      default.tpl
    home/
      index.tpl
```

Resolution order:

1. Host override
2. Module default

This enables:

* theming
* customization
* white-labeling
* UI tweaks without forking modules

---

## Layouts

Layouts are templates that wrap rendered content.

A layout typically:

* renders `<html>`, `<head>`, `<body>`
* includes a content slot
* may define named sections

Example layout responsibilities:

* global CSS / JS
* navigation
* footer

Layouts are **not implicit**.
A view must explicitly choose a layout.

---

## Partials and Fragments

Partials are templates rendered **without layouts**.

Example:

```php
$renderer->render(
    'Skeleton',
    'home/_row',
    ['row' => $row]
);
```

Fragments build on this idea and formalize partial HTML responses.

---

## Fragments and Section-Based Composition

`boru/dweb` supports **explicit fragment rendering** and **section-based layout composition**.
These features are opt-in and do not change routing or controller semantics.

---

### Fragments

A **fragment** is a View that:

* Renders without a layout
* Returns partial HTML
* Is tagged for downstream consumers (HTMX, JS, SSE, etc.)

```php
return $this->fragment('hello');
```

Fragment responses include:

```
X-DWeb-Fragment: Skeleton.hello
```

Fragments are client-agnostic and may be consumed by:

* HTMX
* `fetch()`
* Server-Sent Events (SSE)
* Full-page navigation

---

### SectionBag (Layout Composition)

Views may explicitly populate **named sections** using `SectionBag`.

```php
$this->sectionSet('sidebar', $this->renderPartialToString('partials/sidebar'));
$this->sectionSet('header',  $this->renderPartialToString('partials/header'));
```

Layouts render sections explicitly:

```smarty
<aside>
  {$sections.sidebar|default:'' nofilter}
</aside>

<main>
  {$sections.content|default:$content nofilter}
</main>
```

Key properties:

* No output buffering
* No implicit capture
* Sections are simple HTML strings
* Layout controls placement

---

### Fragment → Section Targeting

Fragments may declare **which section they update**.

```php
return $this->fragmentToSection(
    'sidebar',
    'sidebar',
    ['items' => $items]
);
```

This response includes:

```
X-DWeb-Section: sidebar
```

Optional selector hints may also be included:

```
X-DWeb-Selector: #section-sidebar
X-DWeb-Swap: outerHTML
```

This enables precise UI updates without full page reloads and without framework-side magic.

---

### Section Selector Convention

By default, sections follow a stable DOM convention:

```
section name: sidebar
DOM id:        section-sidebar
selector:      #section-sidebar
```

Helpers are available in templates:

```smarty
{$dweb->sectionId('sidebar')}
{$dweb->sectionSelector('sidebar')}
```

This keeps server-side rendering, layouts, and client-side updates aligned.

---

## Rendering and Actions

Actions generally do **not** render templates.

However:

* Nothing forbids rendering HTML from an Action
* This is useful for HTMX- or SSE-driven UI updates

This is a deliberate design choice.

---

## Error Rendering

HTML error pages are rendered via:

* `ExceptionResponder`
* `ErrorResponse`
* Templates (optionally)

Errors respect:

* content type
* debug flags
* sanitization rules

---

## Design Rules (Important)

* Views control rendering
* Renderer controls output
* Locator controls filesystem access
* Templates contain no logic beyond presentation
* There is no automatic view → template mapping

If a template is rendered, it should be obvious **where** and **why**.

---

## Summary

Rendering in `boru/dweb` is:

* explicit
* layered
* predictable
* override-friendly
* future-proof

The framework provides the structure.
Modules provide the UI.
Renderers provide the implementation.

Nothing happens unless you ask for it.

```