````markdown
# Fun Layers in boru/dweb

Fun layers are **optional capability extensions** built on top of the stable
`boru/dweb` core.

They add ergonomics and UI composition features without changing or weakening
any core guarantees.

Fun layers:

- do **not** modify routing
- do **not** modify dependency injection
- do **not** modify module contracts
- are fully opt-in
- remain explicit and predictable

---

## What Are Fun Layers?

Fun layers sit *above* the core framework and *below* application code.

They exist to:

- improve developer ergonomics
- enable modern UI patterns
- keep controllers and views explicit
- avoid framework-level magic

If a fun layer is not used, nothing changes.

---

## Implemented Fun Layers

### Fragment Rendering

Fragments are layout-free views that return partial HTML.

Characteristics:

- Rendered explicitly by views
- Tagged responses
- Client-agnostic

Fragments are ideal for:

- HTMX
- AJAX / `fetch()`
- Server-Sent Events (SSE)
- Progressive enhancement

Fragments are **not** a separate routing concept—they are regular views rendered without layouts.

---

### SectionBag (Section / Slot Rendering)

`SectionBag` enables **explicit layout composition** using named sections.

Characteristics:

- Sections are simple HTML strings
- No output buffering
- No implicit capture
- Layout controls placement

Views explicitly populate sections:

```php
$this->sectionSet('sidebar', $this->renderPartialToString('partials/sidebar'));
$this->sectionSet('header',  $this->renderPartialToString('partials/header'));
````

Layouts explicitly render sections:

```smarty
<aside>
  {$sections.sidebar|default:'' nofilter}
</aside>

<main>
  {$sections.content|default:$content nofilter}
</main>
```

---

### Fragment → Section Targeting

Fragments may declare **which section they update**.

```php
return $this->fragmentToSection(
    'content',
    'list',
    ['rows' => $rows]
);
```

This response includes metadata such as:

```
X-DWeb-Section: content
X-DWeb-Selector: #section-content
X-DWeb-Swap: outerHTML
```

This enables precise UI updates without full page reloads.

Fragment → section targeting works with:

* HTMX
* custom JavaScript
* SSE-driven updates
* non-JS clients (metadata is simply ignored)

---

### Section Selector Convention

A stable DOM convention is used by default:

```
section name: content
DOM id:        section-content
selector:      #section-content
```

Helpers are available in templates:

```smarty
{$dweb->sectionId('content')}
{$dweb->sectionSelector('content')}
```

This keeps server-side rendering and client-side updates aligned.

---

### HTMX Response Helpers

HTMX support is provided through **explicit response helpers**.

Characteristics:

* Headers are set intentionally
* No auto-detection of HTMX requests
* Safe for non-HTMX clients

Examples include:

* `HX-Retarget`
* `HX-Reswap`
* `HX-Redirect`
* `HX-Trigger`

HTMX helpers are additive and never required.

---

### Canonical URL Helpers

Canonical URL helpers allow templates to generate URLs consistently.

Features:

* Config-driven canonical vs query URLs
* Base-path aware
* No rewrite assumptions

Template helpers:

```smarty
{$dweb->view('home')}
{$dweb->action('echo')}
```

The host application controls canonical behavior via configuration.

---

### AssetManager (Section-Aware)

The rebuilt `AssetManager` integrates directly with `SectionBag`.

Characteristics:

* Assets are registered into sections
* No global state
* No output buffering
* Layout controls final placement

Example:

```smarty
{$dweb->assets()->css('assets/app.css')}
{$dweb->assets()->js('assets/app.js', ['defer' => true])}
```

Layouts decide where assets appear:

```smarty
<head>
  {$dweb->section('head')}
</head>

<body>
  {$content}
  {$dweb->section('scripts')}
</body>
```

---

## Design Principles

All fun layers adhere to the following rules:

* Explicit APIs
* Zero framework magic
* PHP 5.6+ compatibility
* Optional usage
* Client-agnostic behavior

If a fun layer is removed, core rendering still works exactly the same.

---

## Example: Section-Based UI Update

Fragment response:

```php
return $this->fragmentToSection(
    'content',
    'list',
    ['rows' => $rows]
);
```

Layout:

```smarty
<main id="{$dweb->sectionId('content')}">
  {$sections.content nofilter}
</main>
```

HTMX button:

```smarty
<button
  hx-get="{$dweb->view('listFragment')}"
  hx-target="{$dweb->sectionSelector('content')}"
  hx-swap="outerHTML">
  Refresh
</button>
```

---

## Relationship to Core Framework

Fun layers:

* extend capabilities
* do not redefine architecture
* do not change core contracts
* remain easy to reason about

They exist to make common patterns easier—never to hide how the framework works.

---

## Summary

Fun layers in `boru/dweb` provide:

* modern UI composition
* partial rendering
* section-aware layouts
* progressive enhancement support

All while preserving:

* explicit control
* predictable behavior
* long-term maintainability

The core remains small and stable.
Fun layers stay optional and deliberate.

```