<?php
namespace boru\dweb\Kernel;

use boru\dweb\Contracts\ModuleInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Routing\RouteCollection;
use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Mvc\DefaultViewFactory;
use boru\dweb\Mvc\DefaultActionFactory;

class ModuleManager implements \IteratorAggregate
{
    /** @var ModuleInterface[] */
    private $modules = array();

    public function add(ModuleInterface $module)
    {
        $this->modules[] = $module;
        return $this;
    }

    /**
     * Boot modules into container and routes.
     *
     * @param Container $container
     * @param SettingsInterface $settings
     * @return RouteCollection
     */
    public function boot(Container $container, SettingsInterface $settings)
    {
        // Make ModuleManager discoverable for factories (namespace-agnostic module lookup)
        $container->set(__CLASS__, $this);
        $container->set('dweb.module_manager', $this);

        foreach ($this->modules as $m) {
            $this->autoBindModuleFactories($container, $m);
            $m->register($container, $settings);
        }

        $routes = new RouteCollection();

        foreach ($this->modules as $m) {
            $moduleRoutes = $routes->forModule($m->getName());
            $m->routes($moduleRoutes, $container, $settings);
        }

        return $routes;
    }


    private function bindIfClassExists(\boru\dweb\Kernel\Container $c, $id, $factoryCallable)
    {
        if (!class_exists($id)) return;
        // do not overwrite if host/module already bound it
        try {
            $c->get($id);
            return;
        } catch (\Exception $e) {
            // not bound yet: proceed
        }
        $c->set($id, $factoryCallable);
    }

    private function autoBindModuleFactoriesOldWorking(\boru\dweb\Kernel\Container $c, $moduleName)
    {
        $moduleName = (string)$moduleName;

        $viewFactoryClass = 'boru\\dweb\\Modules\\' . $moduleName . '\\Views\\ViewFactory';
        $actionFactoryClass = 'boru\\dweb\\Modules\\' . $moduleName . '\\Actions\\ActionFactory';

        // ViewFactory: known signature (RendererInterface, moduleName)
        $this->bindIfClassExists($c, $viewFactoryClass, function($c) use ($viewFactoryClass, $moduleName) {
            $renderer = $c->get(\boru\dweb\Contracts\RendererInterface::class);
            return new $viewFactoryClass($renderer, $moduleName);
        });

        // ActionFactory: default no-arg (modules can override manually if needed)
        $this->bindIfClassExists($c, $actionFactoryClass, function($c) use ($actionFactoryClass) {
            return new $actionFactoryClass();
        });
    }

    /**
     * Auto-bind view/action factories for a module.
     *
     * - Prefer module-derived namespace (custom module namespaces supported)
     * - Look for module-local factories first:
     *     <ModuleBase>\Views\ViewFactory
     *     <ModuleBase>\Actions\ActionFactory
     * - Fall back to default factories
     *
     * @param Container $container
     * @param \boru\dweb\Contracts\ModuleInterface $module
     * @return void
     */
    private function autoBindModuleFactories(Container $container, $module)
    {
        $moduleName = (string)$module->getName();

        $viewId   = 'dweb.view_factory.' . $moduleName;
        $actionId = 'dweb.action_factory.' . $moduleName;

        // Avoid rebinding if host app already bound them
        // (Container::get throws if missing; no has() assumed)
        try { $container->get($viewId); } catch (\Exception $e) { $this->bindViewFactory($container, $module, $viewId); }
        try { $container->get($actionId); } catch (\Exception $e) { $this->bindActionFactory($container, $module, $actionId); }
    }

    private function bindViewFactory(Container $container, $module, $id)
    {
        // Resolver (optional, but preferred)
        $resolver = null;
        try { $resolver = $container->get(\boru\dweb\Mvc\ModuleNamespaceResolver::class); } catch (\Exception $e) {}

        $baseNs = $resolver ? $resolver->moduleBaseNamespace($module) : $this->deriveBaseNamespaceFromModule($module);
        $customFactoryClass = $baseNs . '\\Views\\ViewFactory';

        if (class_exists($customFactoryClass)) {
            // Support both signatures:
            //   new ViewFactory($renderer, $module, $resolver)
            //   new ViewFactory($renderer, $moduleName)
            $container->set($id, function ($c) use ($customFactoryClass, $module, $resolver) {
                $renderer = $c->get(\boru\dweb\Contracts\RendererInterface::class);

                try {
                    return new $customFactoryClass($renderer, $module, $resolver);
                } catch (\Exception $e) {
                    return new $customFactoryClass($renderer, $module->getName());
                }
            });
            return;
        }

        // Default view factory (module instance + resolver)
        $container->set($id, function ($c) use ($module, $resolver) {
            $renderer = $c->get(\boru\dweb\Contracts\RendererInterface::class);
            return new \boru\dweb\Mvc\DefaultViewFactory($renderer, $module, $resolver);
        });
    }

    private function bindActionFactory(Container $container, $module, $id)
    {
        $resolver = null;
        try { $resolver = $container->get(\boru\dweb\Mvc\ModuleNamespaceResolver::class); } catch (\Exception $e) {}

        $baseNs = $resolver ? $resolver->moduleBaseNamespace($module) : $this->deriveBaseNamespaceFromModule($module);
        $customFactoryClass = $baseNs . '\\Actions\\ActionFactory';

        if (class_exists($customFactoryClass)) {
            // Support signatures:
            //   new ActionFactory($module, $resolver)
            //   new ActionFactory()
            //   new ActionFactory($moduleName)
            $container->set($id, function ($c) use ($customFactoryClass, $module, $resolver) {
                try {
                    return new $customFactoryClass($module, $resolver);
                } catch (\Exception $e) {
                    try {
                        return new $customFactoryClass();
                    } catch (\Exception $e2) {
                        return new $customFactoryClass($module->getName());
                    }
                }
            });
            return;
        }

        // Default action factory (module instance + resolver)
        $container->set($id, function ($c) use ($module, $resolver) {
            $renderer = $c->get(\boru\dweb\Contracts\RendererInterface::class);
            return new \boru\dweb\Mvc\DefaultActionFactory($renderer, $module, $resolver);
        });

    }

    /**
     * Get (and cache) a view factory for a module.
     * Uses module factory if present, else DefaultViewFactory.
     *
     * @param string $moduleName
     * @param \boru\dweb\Kernel\Container $container
     * @return object factory with ->make($viewName)
     */
    public static function getViewFactory($moduleName, $container)
    {
        $moduleName = (string)$moduleName;
        $id = 'dweb.view_factory.' . $moduleName;

        // return cached if set
        try { return $container->get($id); } catch (\Exception $e) {}

        // Try to resolve real module instance (preferred)
        $module = null;
        try {
            $mm = $container->get(\boru\dweb\Kernel\ModuleManager::class);
            $module = $mm->getByName($moduleName);
        } catch (\Exception $e) {
            // ignore; fallback to legacy
        }

        // Resolver (optional service)
        $resolver = null;
        try { $resolver = $container->get(\boru\dweb\Mvc\ModuleNamespaceResolver::class); } catch (\Exception $e) {}

        // Determine where to look for factories
        if ($module) {
            $baseNs = $resolver ? $resolver->moduleBaseNamespace($module) : self::deriveBaseNamespaceFromModule($module);
            $custom = $baseNs . '\\Views\\ViewFactory';
        } else {
            // legacy behavior
            $custom = 'boru\\dweb\\Modules\\' . $moduleName . '\\Views\\ViewFactory';
        }

        $renderer = $container->get(RendererInterface::class);

        if (class_exists($custom)) {
            // Custom factories: try the "new" signature first, then fall back to old.
            if ($module) {
                try {
                    $factory = new $custom($renderer, $module, $resolver);
                } catch (\Exception $e) {
                    $factory = new $custom($renderer, $moduleName);
                }
            } else {
                $factory = new $custom($renderer, $moduleName);
            }
        } else {
            $factory = new DefaultViewFactory($renderer, $module ? $module : $moduleName, $resolver);
        }

        $container->set($id, $factory);
        return $factory;
    }

    /** @return string */
    private static function deriveBaseNamespaceFromModule($module)
    {
        $fqcn = get_class($module);
        $pos = strrpos($fqcn, '\\');
        if ($pos === false) return $fqcn;
        return substr($fqcn, 0, $pos);
    }


    /**
     * Get (and cache) an action factory for a module.
     * Uses module factory if present, else DefaultActionFactory.
     *
     * @param string $moduleName
     * @param \boru\dweb\Kernel\Container $container
     * @return object factory with ->make($actionName)
     */
    public static function getActionFactory($moduleName, $container)
    {
        $moduleName = (string)$moduleName;
        $id = 'dweb.action_factory.' . $moduleName;

        // return cached if set
        try { return $container->get($id); } catch (\Exception $e) {}

        // return cached if set
        try { return $container->get($id); } catch (\Exception $e) {}

        // Try to resolve real module instance (preferred)
        $module = null;
        try {
            $mm = $container->get(\boru\dweb\Kernel\ModuleManager::class);
            $module = $mm->getByName($moduleName);
        } catch (\Exception $e) {
            // ignore; fallback to legacy
        }

        // Resolver (optional service)
        $resolver = null;
        try { $resolver = $container->get(\boru\dweb\Mvc\ModuleNamespaceResolver::class); } catch (\Exception $e) {}

        // Determine where to look for factories
        if ($module) {
            $baseNs = $resolver ? $resolver->moduleBaseNamespace($module) : self::deriveBaseNamespaceFromModule($module);
            $custom = $baseNs . '\\Actions\\ActionFactory';
        } else {
            // legacy behavior
            $custom = 'boru\\dweb\\Modules\\' . $moduleName . '\\Actions\\ActionFactory';
        }

        $renderer = $container->get(RendererInterface::class);

        if (class_exists($custom)) {
            // Still default expectation: no-arg ActionFactory
            // (module can always bind $id manually if it wants DI)
            $factory = new $custom();
        } else {
            $factory = new \boru\dweb\Mvc\DefaultActionFactory($renderer, $module ? $module : $moduleName, $resolver);
        }

        $container->set($id, $factory);
        return $factory;
    }


    /** @return \boru\dweb\Contracts\ModuleInterface[] */
    public function all()
    {
        return $this->modules;
    }

    /** @return \boru\dweb\Contracts\ModuleInterface|null */
    public function getByName($name)
    {
        $want = strtolower((string)$name);
        foreach ($this->modules as $m) {
            if (strtolower($m->getName()) === $want) return $m;
        }
        return null;
    }


    public function getIterator()
    {
        return new \ArrayIterator($this->modules);
    }
}
