<?php
namespace boru\dweb\Mvc;

use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Http\DWebHeaders;
use boru\dweb\Http\HtmlResponse;
use boru\dweb\Http\HxResponse;
use boru\dweb\Rendering\SectionBag;

abstract class AbstractView
{
    /** @var RendererInterface */
    protected $renderer;

    /** @var string */
    protected $moduleName;

    /** @var SectionBag|null */
    private $sections;

    /**
     * @param RendererInterface $renderer
     * @param string $moduleName
     */
    public function __construct(RendererInterface $renderer, $moduleName)
    {
        $this->renderer = $renderer;
        $this->moduleName = (string)$moduleName;
    }

    /**
     * Render a template without layout.
     *
     * @param string $templateId
     * @param array $data
     * @param int $status
     * @param array $headers
     * @return HtmlResponse
     */
    protected function html($templateId, array $data = array(), $status = 200, array $headers = array())
    {
        $html = $this->renderer->render($this->moduleName, $templateId, $data);
        return new HtmlResponse($html, $status, $headers);
    }

    /**
     * Render a template inside a layout (explicit).
     *
     * @param string $layoutId
     * @param string $templateId
     * @param array $data
     * @param int $status
     * @param array $headers
     * @return HtmlResponse
     */
    protected function layout($layoutId, $templateId, array $data = array(), $status = 200, array $headers = array())
    {
        $html = $this->renderer->renderLayout($this->moduleName, $layoutId, $templateId, $data);
        return new HtmlResponse($html, $status, $headers);
    }

    /**
     * Semantic alias for html(): indicates intent that this is a partial render.
     */
    protected function partial($templateId, array $data = array(), $status = 200, array $headers = array())
    {
        return $this->html($templateId, $data, $status, $headers);
    }

    /**
     * Render a fragment (no layout) from templates/fragments/<id>.
     * Tags the response for downstream layers (e.g., HTMX) without magic.
     */
    protected function fragment($fragmentId, array $data = array(), $status = 200, array $headers = array())
    {
        $headers['X-DWeb-Fragment'] = $this->moduleName . '.' . (string)$fragmentId;
        return $this->html('fragments/' . (string)$fragmentId, $data, $status, $headers);
    }

    /**
     * Explicit fragment response with optional HTMX headers.
     *
     * $hx is an associative array of instructions, e.g.:
     *  array(
     *    'retarget' => '#content',
     *    'reswap'   => 'outerHTML',
     *    'pushUrl'  => '/Skeleton/home',
     *    'trigger'  => array('toast' => array('msg' => 'Saved!')),
     *  )
     *
     * @param string $fragmentId
     * @param array $data
     * @param array $hx
     * @param int $status
     * @param array $headers
     * @return \boru\dweb\Http\HtmlResponse
     */
    protected function fragmentHx($fragmentId, array $data = array(), array $hx = array(), $status = 200, array $headers = array())
    {
        $res = $this->fragment($fragmentId, $data, $status, $headers);
        return $this->applyHx($res, $hx);
    }

    /**
     * Apply HTMX headers to any HtmlResponse explicitly.
     * @param \boru\dweb\Http\HtmlResponse $res
     * @param array $hx
     * @return \boru\dweb\Http\HtmlResponse
     */
    protected function applyHx($res, array $hx)
    {
        if (isset($hx['redirect']))  HxResponse::redirect($res, $hx['redirect']);
        if (isset($hx['refresh']))   HxResponse::refresh($res, (bool)$hx['refresh']);
        if (isset($hx['location']))  HxResponse::location($res, $hx['location']);
        if (isset($hx['pushUrl']))   HxResponse::pushUrl($res, $hx['pushUrl']);
        if (isset($hx['replaceUrl']))HxResponse::replaceUrl($res, $hx['replaceUrl']);
        if (isset($hx['reswap']))    HxResponse::reswap($res, $hx['reswap']);
        if (isset($hx['retarget']))  HxResponse::retarget($res, $hx['retarget']);

        if (isset($hx['trigger']))          HxResponse::trigger($res, $hx['trigger'], 'now');
        if (isset($hx['triggerAfterSwap'])) HxResponse::trigger($res, $hx['triggerAfterSwap'], 'afterSwap');
        if (isset($hx['triggerAfterSettle'])) HxResponse::trigger($res, $hx['triggerAfterSettle'], 'afterSettle');

        // optional explicit fragment tag override (mostly for non-view cases)
        if (isset($hx['fragment'])) HxResponse::fragment($res, $hx['fragment']);

        return $res;
    }

    /**
     * Lazily create / reuse section bag for this view invocation.
     * @return SectionBag
     */
    protected function sections()
    {
        if ($this->sections === null) {
            $this->sections = new SectionBag();
        }
        return $this->sections;
    }

    /**
     * Replace a section explicitly.
     * @param string $name
     * @param string $html
     * @return void
     */
    protected function sectionSet($name, $html)
    {
        $this->sections()->set($name, $html);
    }

    /**
     * Append to a section explicitly.
     * @param string $name
     * @param string $html
     * @return void
     */
    protected function sectionAppend($name, $html)
    {
        $this->sections()->append($name, $html);
    }

    /**
     * Render a template to a string (no layout), intended for sections.
     * This is explicit "partial render".
     *
     * @param string $templateId
     * @param array $data
     * @return string
     */
    protected function renderPartialToString($templateId, array $data = array())
    {
        return $this->renderer->render($this->moduleName, (string)$templateId, $data);
    }

    /**
     * Common pattern:
     * - render your main view template into "content"
     * - render sidebar/header/etc into other sections
     * - render layout with $sections available
     *
     * @param string $viewTemplateId e.g. "home/index"
     * @param array $data
     * @param string $layoutId e.g. "layout/main" (whatever your renderer expects)
     * @param int $status
     * @param array $headers
     * @return HtmlResponse
     */
    protected function htmlWithSections($viewTemplateId, array $data, $layoutId, $status = 200, array $headers = array())
    {
        $content = $this->renderer->render($this->moduleName, (string)$viewTemplateId, $data);

        // Ensure a default "content" section exists
        $sections = $this->sections();
        $sections->ensure('content', $content);

        // Layout gets both:
        // - $content (back-compat / simplest)
        // - $sections (new slot system)
        $layoutData = $data;
        $layoutData['content']  = $sections->get('content');
        $layoutData['sections'] = $sections->all();

        // Assumes you already have a layout render method in renderer:
        // renderLayout($moduleName, $layoutId, $layoutData)
        $html = $this->renderer->render($this->moduleName, (string)$layoutId, $layoutData);

        return new HtmlResponse($html, $status, $headers);
    }

    /**
     * Default section selector convention used by this view.
     * Override in a child view if you want a different convention.
     *
     * @param string $sectionName
     * @return string
     */
    protected function sectionSelector($sectionName)
    {
        return '#section-' . (string)$sectionName;
    }

    /**
     * Fragment -> Section targeting (explicit + convenient):
     * - Always annotates X-DWeb-Section
     * - Optionally auto-applies HX-Retarget/HX-Reswap using the section selector convention
     * - Caller can override by providing 'retarget'/'reswap' in $hx or custom $selector/$swap
     *
     * @param string $sectionName
     * @param string $fragmentId
     * @param array  $data
     * @param array  $hx
     * @param int    $status
     * @param array  $headers
     * @param string|null $selector If null, uses sectionSelector($sectionName)
     * @param string|null $swap If null, defaults to 'outerHTML'
     * @param bool $applyHxTargeting If true, will set HX headers when not already set
     * @return \boru\dweb\Http\HtmlResponse
     */
    protected function fragmentToSection(
        $sectionName,
        $fragmentId,
        array $data = array(),
        array $hx = array(),
        $status = 200,
        array $headers = array(),
        $selector = null,
        $swap = null,
        $applyHxTargeting = true
    ) {
        $sectionName = (string)$sectionName;

        // Always annotate which section this fragment belongs to
        $headers[DWebHeaders::SECTION] = $sectionName;

        // Convention selector + default swap
        if ($selector === null) $selector = $this->sectionSelector($sectionName);
        if ($swap === null) $swap = 'outerHTML';

        // Optional: include generic hints for any clients
        $headers[DWebHeaders::SELECTOR] = (string)$selector;
        $headers[DWebHeaders::SWAP]     = (string)$swap;

        // Optional: apply HTMX targeting headers (still explicit, but convenient)
        if ($applyHxTargeting) {
            if (!isset($hx['retarget'])) $hx['retarget'] = (string)$selector;
            if (!isset($hx['reswap']))   $hx['reswap']   = (string)$swap;
        }

        // Leverage your existing fragmentHx() (already tags fragment + HX headers)
        return $this->fragmentHx($fragmentId, $data, $hx, $status, $headers);
    }
}
