<?php
namespace boru\dweb\Mvc;

use boru\dweb\Contracts\ModuleInterface;
use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Exceptions\Fail;

class DefaultActionFactory
{
    /** @var RendererInterface */
    private $renderer;

    /** @var string */
    private $moduleName;

    /** @var string */
    private $moduleNamespace;

    /** @var ModuleInterface|null */
    private $module;

    /** @var ModuleNamespaceResolver|null */
    private $resolver;

    /**
     * @param RendererInterface $renderer
     * @param string $moduleName
     */
    public function __construct(RendererInterface $renderer, $moduleOrName, ModuleNamespaceResolver $resolver = null)
    {
        $this->renderer = $renderer;
        $this->resolver = $resolver;
        if ($moduleOrName instanceof ModuleInterface) {
            $this->module = $moduleOrName;
            $this->moduleName = (string)$moduleOrName->getName();

            // derive base namespace from module class (preferred)
            if ($resolver) {
                $this->moduleNamespace = $resolver->moduleBaseNamespace($moduleOrName);
            } else {
                // fallback: same derivation without resolver
                $fqcn = get_class($moduleOrName);
                $pos = strrpos($fqcn, '\\');
                $this->moduleNamespace = ($pos === false) ? $fqcn : substr($fqcn, 0, $pos);
            }
        } else {
            // legacy behavior: assume boru\dweb\Modules\<Module>
            $this->module = null;
            $this->moduleName = (string)$moduleOrName;
            $this->moduleNamespace = 'boru\\dweb\\Modules\\' . $this->moduleName;
        }
    }

    /**
     * Create an Action by local name (e.g. "ping" => PingAction).
     *
     * Convention: <Module>\Actions\<StudlyName>Action
     *
     * @param string $actionName
     * @return object
     */
    public function make($actionName)
    {
        $class = $this->moduleNamespace . '\\Actions\\' . $this->studly($actionName) . 'Action';
        if (!class_exists($class)) {
            $excption = Fail::runtime("Action class not found: " . $class, array(
                'module' => $this->moduleName,
                'action' => (string)$actionName,
                'namespace' => $this->moduleNamespace,
            ));
            throw $excption;
        }

        return $this->instantiateAction($class);
    }

    /**
     * Instantiate safely across PHP 5.6 - 8.x.
     * Avoids fatal ArgumentCountError by using Reflection.
     *
     * Rules:
     * - If it extends AbstractAction: prefer ctor($renderer, $moduleName) when compatible
     * - Else: prefer no-arg when possible
     * - Else: best-effort with ($renderer) or ($renderer, $moduleName) if ctor allows
     *
     * @param string $class
     * @return object
     */
    private function instantiateAction($class)
    {
        $ref = new \ReflectionClass($class);
        $ctor = $ref->getConstructor();

        if ($ctor === null) {
            return $ref->newInstance();
        }

        $required = $ctor->getNumberOfRequiredParameters();
        $total    = $ctor->getNumberOfParameters();

        $isAbstractAction = is_subclass_of($class, __NAMESPACE__ . '\\AbstractAction');

        // Preferred: AbstractAction-like signature: (RendererInterface $r, string $moduleName)
        if ($isAbstractAction) {
            if ($required <= 2 && $total >= 2) {
                return $ref->newInstanceArgs(array($this->renderer, $this->moduleName));
            }
            if ($required <= 1 && $total >= 1) {
                return $ref->newInstanceArgs(array($this->renderer));
            }
            if ($required === 0) {
                return $ref->newInstance();
            }

            throw new \RuntimeException(
                "Action constructor not supported for {$class}: requires {$required} params"
            );
        }

        // Non-AbstractAction actions (legacy/custom)
        if ($required === 0) {
            return $ref->newInstance();
        }
        if ($required <= 1 && $total >= 1) {
            return $ref->newInstanceArgs(array($this->renderer));
        }
        if ($required <= 2 && $total >= 2) {
            return $ref->newInstanceArgs(array($this->renderer, $this->moduleName));
        }

        throw new \RuntimeException(
            "Action constructor not supported for {$class}: requires {$required} params"
        );
    }

    private function studly($name)
    {
        $name = preg_replace('/[^a-zA-Z0-9_]/', '_', (string)$name);
        $parts = preg_split('/_+/', $name);
        $out = '';
        foreach ($parts as $p) {
            if ($p === '') continue;
            $out .= strtoupper(substr($p, 0, 1)) . substr($p, 1);
        }
        return $out;
    }
}
