<?php
namespace boru\dweb;

use boru\dweb\Config\ConfigKeys;
use boru\dweb\Contracts\ModuleInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Http\Request;
use boru\dweb\Http\Emitter;
use boru\dweb\Routing\WebRouter;
use boru\dweb\Routing\PathRouter;
use boru\dweb\Kernel\Container;
use boru\dweb\Kernel\ModuleManager;
use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Contracts\TemplateLocatorInterface;
use boru\dweb\Rendering\TemplateLocator;
use boru\dweb\Contracts\MiddlewareInterface;
use boru\dweb\Contracts\LoggerInterface;
use boru\dweb\Logging\NullLogger;
use boru\dweb\Logging\ErrorLogLogger;
use boru\dweb\Assets\AssetPublisherInterface;
use boru\dweb\Assets\FilesystemAssetPublisher;
use boru\dweb\Kernel\Services;

class WebUI
{

    private $env;

    /** @var Container|null */
    private $container = null;

    private $routes;
    private $router;
    private $pipeline;
    private $containerBootstrapCallback;

    /** @var SettingsInterface */
    private $config;

    /** @var ModuleManager */
    private $modules;

    /** @var MiddlewareInterface[] */
    private $middleware = [];

    /** @var callable[] */
    private $containerBootstraps = [];

    public function __construct(SettingsInterface $config)
    {
        $this->config = $config;
        $this->modules = new ModuleManager();
    }

    public function withModule(ModuleInterface $module)
    {
        $this->modules->add($module);
        return $this;
    }

    /**
     * Host app hook: configure container before modules boot.
     *
     * Signature:
     *   function(Container $c, SettingsInterface $settings): void
     *
     * @param callable $bootstrap
     * @return $this
     */
    public function withContainerBootstrap($bootstrap)
    {
        $this->containerBootstraps[] = $bootstrap;
        return $this;
    }

    /**
     * Add global middleware to the app.
     *
     * @param MiddlewareInterface $mw
     * @return $this
     */
    public function withMiddleware(\boru\dweb\Contracts\MiddlewareInterface $mw)
    {
        $this->middleware[] = $mw;
        return $this;
    }

    private function boot() {
        $spec = \boru\dweb\Support\KernelSpec::fromConfig($this->config)
        ->withContainer($this->containerBootstrapCallback)
        ->withModules(function ($mm, $c, $cfg) {
            foreach ($this->modules as $m) $mm->add($m);
        })
        ->withMiddleware($this->middleware);

        $this->env = \boru\dweb\Support\KernelBuilder::buildEnv($spec);
    }

    private function bootOld() {
        $modules = $this->modules->all(); // capture for clarity

        list($container, $routes, $router, $pipeline) = \boru\dweb\Support\KernelBuilder::build(
            $this->config,
            $this->containerBootstrapCallback, // if you have one
            function ($mm, $c, $cfg) use ($modules) {
                // apply $modules list into $mm
                foreach ($modules as $m) $mm->add($m);
            },
            $this->middleware // if host adds middleware via WebUI
        );

        $this->container = $container;
        $this->routes = $routes;
        $this->router = $router;
        $this->pipeline = $pipeline;
    }

    public function getContainer()
    {
        if (!$this->env) $this->boot();
        return $this->env->container();
    }
    public function getContainerOld()
    {
        if (!$this->container) {
            $this->boot();
        }
        return $this->container;
    }

    public function render()
    {
        $req = null;
        try {
            if (!$this->env) $this->boot();

            $req = $this->env->container()->get(\boru\dweb\Http\Request::class);
            $response = call_user_func($this->env->pipeline(), $req);
        } catch (\Exception $e) {
            // Try to log centrally (best effort)
            try {
                if($this->env && $this->env->container()) {
                    $logger = $this->env->container()->get(\boru\dweb\Contracts\LoggerInterface::class);
                    $logger->error('Unhandled exception', array(
                        'exception' => get_class($e),
                        'message' => $e->getMessage(),
                        'file' => $e->getFile(),
                        'line' => $e->getLine(),
                    ));
                }
            } catch (\Exception $ignore) {}

            if ($req === null) {
                try { $req = \boru\dweb\Http\Request::fromGlobals(); } catch (\Exception $ignore) {}
            }
            $debug = (bool)$this->config->get(ConfigKeys::DEBUG_ENABLED, false);
            $response = \boru\dweb\Http\ExceptionResponder::fromException($e, $req, $debug, $this->config);
        }

        \boru\dweb\Http\Emitter::emit($response);
    }
}
