# Writing a dweb Module

This document explains how to author modules for `boru/dweb`.

---

## Module Structure

Example: `Skeleton` module

````

Modules/
Skeleton/
SkeletonModule.php
Views/
HomeView.php
Actions/
EchoAction.php
DeleteThingAction.php
templates/
layouts/
default.tpl
home/
index.tpl

````

---

## Module Interface

Every module implements:

```php
ModuleInterface {
    getName();
    register(Container $container, SettingsInterface $settings);
    routes(ModuleRouteCollection $routes, Container $container, SettingsInterface $settings);
}
````

---

## Registering Services

Use `register()` to bind services:

```php
public function register(Container $c, SettingsInterface $settings)
{
    $c->set(MyService::class, function($c) {
        return new MyService();
    });
}
```

---

## Registering Routes

Routes are registered via `ModuleRouteCollection`.

### Views

```php
$routes->addModuleView('home', $container);
```

Registers:

```
Skeleton.home
```

---

### Actions

```php
$routes->addModuleAction('echo', $container, ['POST']);
```

Registers:

```
Skeleton.echo
```

---

## Views

Views:

* Return HTML
* Use RendererInterface
* Are layout-aware

Example:

```php
class HomeView
{
    public function handle($req)
    {
        return new HtmlResponse(
            $this->renderer->renderLayout(
                'Skeleton',
                'layouts/default',
                'home/index',
                ['title' => 'Hello']
            )
        );
    }
}
```

---

## Actions

Actions:

* Return JSON / SSE
* Are method-restricted
* Typically extend AbstractAction

Example:

```php
class EchoAction extends AbstractAction
{
    public function handle($req)
    {
        return $this->success([
            'name' => $req->param('name')
        ]);
    }
}
```

## Views vs Actions

Views and Actions are semantically different but technically equivalent.

Both may:
- render templates
- return fragments
- return JSON
- return SSE streams

Choose:
- Views for page-oriented controllers
- Actions for API / side-effect controllers

---

## Templates

Templates are:

* Module-scoped
* Located via TemplateLocator
* Overridable by the host app

Resolution order:

1. Custom override
2. Module default

---

## Factories

Modules do NOT need to define factories.

Default factories are provided by:

* `ModuleManager::getViewFactory()`
* `ModuleManager::getActionFactory()`

Custom factories are optional.

---

## Middleware in Modules

Modules do NOT own middleware.

Middleware is added by:

* Host app
* WebUI

This keeps module behavior predictable.

---

## Testing Modules

Modules should be testable via:

* Router-level smoke tests
* WebUI end-to-end smoke tests

Skeleton module exists as:

* Reference
* Validation harness
* Documentation by example

---

## Design Rules (Important)

* No globals
* No direct settings access in controllers
* No framework routes
* No magic wiring
* No implicit behavior

If something requires explanation, prefer making it explicit in code.


