# Middleware in boru/dweb

Middleware in `boru/dweb` is **explicit, ordered, and framework-level**.
There is no implicit middleware, no global stacks, and no module-owned middleware.

This document explains how middleware works and how it should be used.

---

## What Middleware Is

Middleware is code that runs **around request handling**.

It can:
- inspect or modify the request
- short-circuit the request
- modify the response
- enforce cross-cutting concerns

Middleware always wraps the router and applies to:
- Views
- Actions
- Debug endpoints

---

## What Middleware Is NOT

- Middleware is not routing
- Middleware is not business logic
- Middleware is not module behavior
- Middleware is not controller logic

Modules do **not** define middleware.

---

## Middleware Contract

All middleware implements:

```php
MiddlewareInterface {
    public function handle(Request $req, callable $next);
}
````

Middleware:

* receives a `Request`
* must call `$next($req)` to continue
* must return a `Response`

---

## Execution Order

Middleware is executed **in the order it is added**.

Example:

```php
$ui->withMiddleware(new MethodOverrideMiddleware());
$ui->withMiddleware(new JsonBodyMiddleware());
$ui->withMiddleware(new SecurityHeadersMiddleware());
```

Execution order:

```
Request
 → MethodOverrideMiddleware
   → JsonBodyMiddleware
     → SecurityHeadersMiddleware
       → Router
     ← SecurityHeadersMiddleware
   ← JsonBodyMiddleware
 ← MethodOverrideMiddleware
Response
```

Earlier middleware:

* sees the request first
* sees the response last

---

## Where Middleware Is Added

Middleware is added by the **host application**, typically in `public/index.php`.

Example:

```php
$ui->withMiddleware(new JsonBodyMiddleware());
```

The framework does not auto-register middleware.

---

## Shipped Middleware (Documentation by Example)

### MethodOverrideMiddleware

Allows HTTP method override via:

* `_method` POST parameter
* `X-HTTP-Method-Override` header

Purpose:

* support browsers and forms
* enable REST-style APIs

---

### JsonBodyMiddleware

* Parses `application/json` request bodies
* Decodes JSON into an array
* Merges decoded values into request parameters

Benefits:

* Actions can use `$req->param()` uniformly
* No JSON parsing logic in controllers
* Clear error handling on invalid JSON

---

### SecurityHeadersMiddleware

Adds common security headers:

* `X-Content-Type-Options`
* `X-Frame-Options`
* `Referrer-Policy`
* Optional `Content-Security-Policy`

Headers are:

* added only if missing
* configurable via settings

---

## Middleware and Errors

If middleware throws an exception:

* it is handled by `ExceptionResponder`
* content type is respected
* debug flags apply

Middleware should:

* throw `HttpException` for client errors
* avoid catching exceptions unless transforming them

---

## Middleware and Rendering

Middleware runs **before rendering** and **after rendering**.

Typical patterns:

* authentication checks
* request mutation
* response header injection
* logging

Middleware should not:

* render templates
* return HTML unless intentionally short-circuiting

---

## Short-Circuiting Requests

Middleware may return a response without calling `$next()`.

Example use cases:

* authentication failure
* maintenance mode
* rate limiting

This is allowed and intentional.

---

## Middleware vs Controllers

| Concern        | Middleware | Controller |
| -------------- | ---------- | ---------- |
| Authentication | ✔          | ✖          |
| JSON parsing   | ✔          | ✖          |
| Business logic | ✖          | ✔          |
| Rendering      | ✖          | ✔          |
| Headers        | ✔          | ✖          |

---

## Design Rules (Important)

* Middleware is global
* Order matters
* No hidden registration
* No module-level middleware
* Explicit is better than clever

If middleware behavior is surprising, it is incorrectly placed.

---

## Summary

Middleware in `boru/dweb` is:

* explicit
* predictable
* framework-owned
* testable
* minimal

It exists to solve cross-cutting concerns cleanly,
without leaking complexity into modules.
