<?php
namespace boru\dweb\Assets;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\LoggerInterface;
use boru\dweb\Contracts\AssetProviderInterface;
use boru\dweb\Kernel\ModuleManager;
use boru\dweb\Config\ConfigKeys;

class FilesystemAssetPublisher implements AssetPublisherInterface
{
    /** @var SettingsInterface */
    private $settings;

    /** @var ModuleManager */
    private $modules;

    /** @var LoggerInterface */
    private $logger;

    public function __construct(SettingsInterface $settings, ModuleManager $modules, LoggerInterface $logger)
    {
        $this->settings = $settings;
        $this->modules  = $modules;
        $this->logger   = $logger;
    }

    public function publishAll()
    {
        $res = new PublishResult();

        $enabled = (bool)$this->settings->get(ConfigKeys::ASSET_PUBLISH_ENABLED, false);
        if (!$enabled) {
            $res->addMessage('Asset publishing disabled (ConfigKeys::ASSET_PUBLISH_ENABLED).');
            return $res;
        }

        $publicDir = (string)$this->settings->get(ConfigKeys::ASSET_PUBLIC_DIR, '');
        if ($publicDir === '') {
            $res->errors++;
            $res->addMessage('Missing ConfigKeys::ASSET_PUBLIC_DIR (absolute path recommended).');
            return $res;
        }

        $publicDir = rtrim($publicDir, DIRECTORY_SEPARATOR);

        // Ensure base dir exists
        if (!is_dir($publicDir) && !@mkdir($publicDir, 0777, true)) {
            $res->errors++;
            $res->addMessage('Failed to create public assets dir: ' . $publicDir);
            return $res;
        }

        foreach ($this->modules->all() as $module) {
            if (!($module instanceof AssetProviderInterface)) {
                continue;
            }

            $moduleName = $module->getName();
            $srcDir = $module->getAssetPath();
            if ($srcDir === null || $srcDir === '') continue;

            $srcDir = rtrim($srcDir, DIRECTORY_SEPARATOR);
            if (!is_dir($srcDir)) {
                $res->skipped++;
                $res->addMessage("Skipped {$moduleName}: asset dir not found: {$srcDir}");
                continue;
            }

            $destDir = $publicDir . DIRECTORY_SEPARATOR . $moduleName;

            $this->logger->info("Publishing assets for {$moduleName}: {$srcDir} -> {$destDir}");
            $this->copyDir($srcDir, $destDir, $res);
        }

        return $res;
    }

    private function copyDir($src, $dest, PublishResult $res)
    {
        if (!is_dir($dest) && !@mkdir($dest, 0777, true)) {
            $res->errors++;
            $res->addMessage('Failed to create dir: ' . $dest);
            return;
        }

        $dh = @opendir($src);
        if (!$dh) {
            $res->errors++;
            $res->addMessage('Failed to open dir: ' . $src);
            return;
        }

        while (($name = readdir($dh)) !== false) {
            if ($name === '.' || $name === '..') continue;

            $srcPath  = $src  . DIRECTORY_SEPARATOR . $name;
            $destPath = $dest . DIRECTORY_SEPARATOR . $name;

            if (is_dir($srcPath)) {
                $this->copyDir($srcPath, $destPath, $res);
                continue;
            }

            // Copy if missing or changed by mtime/size
            $shouldCopy = true;
            if (is_file($destPath)) {
                $shouldCopy = (
                    @filesize($srcPath) !== @filesize($destPath)
                    || @filemtime($srcPath) > @filemtime($destPath)
                );
            }

            if (!$shouldCopy) {
                $res->skipped++;
                continue;
            }

            // Ensure dest dir exists
            $parent = dirname($destPath);
            if (!is_dir($parent) && !@mkdir($parent, 0777, true)) {
                $res->errors++;
                $res->addMessage('Failed to create parent dir: ' . $parent);
                continue;
            }

            if (!@copy($srcPath, $destPath)) {
                $res->errors++;
                $res->addMessage('Failed to copy: ' . $srcPath . ' -> ' . $destPath);
                continue;
            }

            $res->copied++;
        }

        closedir($dh);
    }
}
