<?php
namespace boru\dweb\Cli;

use boru\dweb\Support\KernelEnv;
use boru\dweb\Cli\Commands\HelpCommand;
use boru\dweb\Cli\Commands\PublishCommand;
use boru\dweb\Cli\Commands\SmokeTestCommand;

class App
{
    /** @var CommandRegistry */
    private $registry;

    public function __construct(CommandRegistry $registry = null)
    {
        $this->registry = $registry ? $registry : new CommandRegistry();
    }

    /**
     * @param array $argv
     * @return int
     */
    public function run(array $argv)
    {
        $args = array_slice($argv, 1);
        $cmdName = isset($args[0]) ? $args[0] : 'help';

        // help shortcuts
        if ($cmdName === '-h' || $cmdName === '--help') $cmdName = 'help';

        // args after command
        array_shift($args);
        $parsed = Args::parse($args);

        // Build core registry first (help always available)
        $env = null;
        $this->registry = $this->buildRegistry($env);

        // Unknown command? show help
        if (!$this->registry->get($cmdName)) {
            $this->err("Unknown command: {$cmdName}\n\n");
            $this->registry->get('help')->execute(array());
            return 2;
        }

        // Help does not require env
        if ($cmdName === 'help') {
            return $this->registry->get('help')->execute($args);
        }

        // Everything else requires an env
        try {
            $env = CliEnv::build($parsed);
        } catch (\Exception $e) {
            $this->err("ERROR: " . $e->getMessage() . "\n\n");
            $this->err("Tip: dweb help\n");
            return 2;
        }

        // Rebuild registry with env-aware commands
        $this->registry = $this->buildRegistry($env);

        // Optional hook: allow env-provided registry extensions
        // (Safe no-op if not bound)
        $this->maybeExtendRegistryFromEnv($env);

        $command = $this->registry->get($cmdName);
        if (!$command) {
            // This can happen if the command is only provided by env/modules
            $this->err("Unknown command: {$cmdName}\n\n");
            $this->registry->get('help')->execute(array());
            return 2;
        }

        return $command->execute($args);
    }

    /**
     * @param KernelEnv|null $env
     * @return CommandRegistry
     */
    private function buildRegistry($env)
    {
        $r = new CommandRegistry();

        $r->add(new HelpCommand($this));

        // Core commands become env-aware when env exists; they can self-error if env is null
        $r->add(new SmokeTestCommand($env));
        $r->add(new PublishCommand($env));

        return $r;
    }

    /**
     * Optional: let host env register commands into a registry service.
     *
     * Pattern (if you choose):
     *   container->set(CommandRegistry::class, $existingRegistry)
     *   modules add commands during register()/commands()
     */
    private function maybeExtendRegistryFromEnv(KernelEnv $env)
    {
        try {
            $c = $env->container();

            // If you bind a CommandRegistry in container, merge it into ours.
            $extra = $c->get(\boru\dweb\Cli\CommandRegistry::class);
            if ($extra instanceof \boru\dweb\Cli\CommandRegistry) {
                foreach ($extra->all() as $cmd) {
                    $this->registry->add($cmd);
                }
            }
        } catch (\Exception $ignore) {
            // no-op
        }
    }

    /** @return CommandInterface[] */
    public function listCommands()
    {
        return $this->registry->all();
    }

    public function out($s) { echo $s; }
    public function err($s) { fwrite(STDERR, $s); }
}
