<?php
namespace boru\dweb\Cli;

use boru\dweb\Support\KernelEnv;
use boru\dweb\Contracts\SettingsInterface;

class CliEnv
{
    /**
     * Build KernelEnv from CLI flags.
     *
     * Supported:
     *  --env-bootstrap=/abs/path/file.php   (must return KernelEnv)
     *  --setEnv=KEY=VALUE                  (repeatable)
     *
     * @param Args $args
     * @return KernelEnv
     */
    public static function build(Args $args)
    {
        $bootstrapPath = $args->get('env-bootstrap', null);
        if (!$bootstrapPath) {
            throw new \RuntimeException("Missing required flag: --env-bootstrap=/path/to/env.php (must return KernelEnv)");
        }

        $env = self::loadEnvBootstrap($bootstrapPath);

        // Apply overrides to SettingsInterface (config)
        $sets = $args->getAll('setEnv'); // repeatable
        foreach ($sets as $expr) {
            self::applySetEnv($env, $expr);
        }

        return $env;
    }

    /**
     * @param string $path
     * @return KernelEnv
     */
    private static function loadEnvBootstrap($path)
    {
        $path = (string)$path;

        if (!file_exists($path)) {
            throw new \RuntimeException("env-bootstrap file not found: " . $path);
        }

        $env = require $path;

        if (!$env instanceof KernelEnv) {
            $t = is_object($env) ? get_class($env) : gettype($env);
            throw new \RuntimeException("env-bootstrap must return boru\\dweb\\Support\\KernelEnv; got: " . $t);
        }

        return $env;
    }

    /**
     * --setEnv=KEY=VALUE
     *
     * @param KernelEnv $env
     * @param string $expr
     * @return void
     */
    private static function applySetEnv(KernelEnv $env, $expr)
    {
        $expr = (string)$expr;
        $pos = strpos($expr, '=');
        if ($pos === false) {
            throw new \RuntimeException("Invalid --setEnv, expected KEY=VALUE, got: " . $expr);
        }

        $key = trim(substr($expr, 0, $pos));
        $raw = trim(substr($expr, $pos + 1));

        if ($key === '') {
            throw new \RuntimeException("Invalid --setEnv, empty key in: " . $expr);
        }

        $value = self::parseValue($raw);

        $cfg = $env->config(); // SettingsInterface-conforming object
        $cfg->set($key, $value);
    }

    /**
     * Parse string to best-effort scalar/JSON.
     *
     * Rules:
     *  - "true"/"false"/"null" => boolean/null
     *  - numeric => int/float
     *  - json object/array => decoded
     *  - otherwise => string (unquoted)
     *
     * @param string $raw
     * @return mixed
     */
    private static function parseValue($raw)
    {
        $raw = (string)$raw;

        // Strip wrapping quotes: "x" or 'x'
        if (strlen($raw) >= 2) {
            $a = $raw[0];
            $b = $raw[strlen($raw) - 1];
            if (($a === '"' && $b === '"') || ($a === "'" && $b === "'")) {
                $raw = substr($raw, 1, -1);
            }
        }

        $lc = strtolower($raw);
        if ($lc === 'true') return true;
        if ($lc === 'false') return false;
        if ($lc === 'null') return null;

        // JSON object/array
        $first = strlen($raw) ? $raw[0] : '';
        if ($first === '{' || $first === '[') {
            $decoded = json_decode($raw, true);
            if (json_last_error() === JSON_ERROR_NONE) return $decoded;
            // fallthrough to string if invalid JSON
        }

        // numeric
        if (is_numeric($raw)) {
            // int vs float
            return (strpos($raw, '.') === false) ? (int)$raw : (float)$raw;
        }

        return $raw;
    }
}
