<?php
namespace boru\dweb\Http;

class HttpException extends \RuntimeException
{
    /** @var int */
    protected $statusCode;

    /** @var string */
    protected $userMessage;

    /** @var string|null */
    protected $debugDetails;

    /**
     * @param int $statusCode
     * @param string $userMessage
     * @param string|null $debugDetails
     * @param int $code
     * @param \Exception|null $previous
     */
    public function __construct($statusCode, $userMessage, $debugDetails = null, $code = 0, \Exception $previous = null)
    {
        $this->statusCode = (int)$statusCode;
        $this->userMessage = (string)$userMessage;
        $this->debugDetails = $debugDetails !== null ? (string)$debugDetails : null;

        parent::__construct($this->userMessage, (int)$code, $previous);
    }

    /** @return int */
    public function getStatusCode()
    {
        return $this->statusCode;
    }

    /** @return string */
    public function getUserMessage()
    {
        return $this->userMessage;
    }

    /** @return string|null */
    public function getDebugDetails()
    {
        return $this->debugDetails;
    }

    /* =========================================================
     * Tiny helpers
     * ========================================================= */

    public static function badRequest($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(400, $userMessage, $debugDetails, 0, $previous);
    }

    public static function unauthorized($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(401, $userMessage, $debugDetails, 0, $previous);
    }

    public static function forbidden($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(403, $userMessage, $debugDetails, 0, $previous);
    }

    public static function notFound($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(404, $userMessage, $debugDetails, 0, $previous);
    }

    public static function conflict($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(409, $userMessage, $debugDetails, 0, $previous);
    }

    public static function unprocessableEntity($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(422, $userMessage, $debugDetails, 0, $previous);
    }

    public static function serviceUnavailable($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(503, $userMessage, $debugDetails, 0, $previous);
    }

    public static function internalError($userMessage, $debugDetails = null, \Exception $previous = null)
    {
        return new self(500, $userMessage, $debugDetails, 0, $previous);
    }

    public static function wrap($statusCode, \Exception $previous, $prefix = null, $debugDetails = null)
    {
        $msg = $previous->getMessage();
        if ($prefix) $msg = (string)$prefix . $msg;
        if ($debugDetails === null) $debugDetails = $previous->getMessage();

        return new self((int)$statusCode, $msg, $debugDetails, 0, $previous);
    }

    /* =========================================================
     * Sanitization
     * ========================================================= */

    public static function sanitizeMessage($message)
    {
        $message = (string)$message;

        // Windows drive paths: C:\dir\dir\file.php or C:/dir/dir/file.php
        $message = preg_replace_callback(
            '/([A-Za-z]:[\\\\\\/](?:[^\\s"\'<>:]+[\\\\\\/])+(?:[^\\s"\'<>:]+))/',
            array(__CLASS__, 'replacePathMatch'),
            $message
        );

        // Unix paths: /dir/dir/file.php
        $message = preg_replace_callback(
            '/(\\/(?:[^\\s"\'<>:]+\\/)+(?:[^\\s"\'<>:]+))/',
            array(__CLASS__, 'replacePathMatch'),
            $message
        );

        return $message;
    }

    private static function replacePathMatch(array $m)
    {
        $full = $m[1];

        $norm = str_replace('\\', '/', $full);
        $parts = explode('/', trim($norm, '/'));

        $n = count($parts);
        if ($n <= 0) return '[path]';

        $base = $parts[$n - 1];
        $parent = ($n >= 2) ? $parts[$n - 2] : null;

        if ($parent && $parent !== '' && $parent !== '.' && $parent !== '..') {
            return '[path]/' . $parent . '/' . $base;
        }

        return '[path]/' . $base;
    }
}
