<?php
namespace boru\dweb\Mvc;

use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Contracts\ModuleInterface;

class DefaultViewFactory
{
    /** @var RendererInterface */
    private $renderer;

    /** @var string */
    private $moduleName;

    /** @var string */
    private $moduleNamespace;

    /** @var ModuleInterface|null */
    private $module;

    /** @var ModuleNamespaceResolver|null */
    private $resolver;

    /**
     * Backward compatible:
     * - Old usage: new DefaultViewFactory($renderer, 'Skeleton')
     * - New usage: new DefaultViewFactory($renderer, $moduleInstance, $resolver)
     *
     * @param RendererInterface $renderer
     * @param string|ModuleInterface $moduleOrName
     * @param ModuleNamespaceResolver|null $resolver
     */
    public function __construct(RendererInterface $renderer, $moduleOrName, ModuleNamespaceResolver $resolver = null)
    {
        $this->renderer = $renderer;
        $this->resolver = $resolver;

        if ($moduleOrName instanceof ModuleInterface) {
            $this->module = $moduleOrName;
            $this->moduleName = (string)$moduleOrName->getName();

            // derive base namespace from module class (preferred)
            if ($resolver) {
                $this->moduleNamespace = $resolver->moduleBaseNamespace($moduleOrName);
            } else {
                // fallback: same derivation without resolver
                $fqcn = get_class($moduleOrName);
                $pos = strrpos($fqcn, '\\');
                $this->moduleNamespace = ($pos === false) ? $fqcn : substr($fqcn, 0, $pos);
            }
        } else {
            // legacy behavior: assume boru\dweb\Modules\<Module>
            $this->module = null;
            $this->moduleName = (string)$moduleOrName;
            $this->moduleNamespace = 'boru\\dweb\\Modules\\' . $this->moduleName;
        }
    }

    /**
     * Create a View by local name (e.g. "home" => HomeView).
     *
     * Convention: <Module>\Views\<StudlyName>View
     *
     * @param string $viewName
     * @return object
     */
    public function make($viewName)
    {
        $class = $this->moduleNamespace . '\\Views\\' . $this->studly($viewName) . 'View';
        if (!class_exists($class)) {
            throw new \RuntimeException("View class not found: " . $class);
        }

        // Default contract: views accept RendererInterface (explicit)
        return new $class($this->renderer, $this->moduleName);
    }

    private function studly($name)
    {
        $name = preg_replace('/[^a-zA-Z0-9_]/', '_', (string)$name);
        $parts = preg_split('/_+/', $name);
        $out = '';
        foreach ($parts as $p) {
            if ($p === '') continue;
            $out .= strtoupper(substr($p, 0, 1)) . substr($p, 1);
        }
        return $out;
    }
}
