<?php
namespace boru\dweb\Rendering;

use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\TemplateLocatorInterface;
use boru\dweb\Http\HttpException;
use boru\dweb\Rendering\DWebSmarty;
use boru\dweb\Routing\CanonicalUrl;

class SmartyRenderer implements RendererInterface
{
    /** @var \DWebSmarty */
    private $smarty;

    /** @var TemplateLocatorInterface */
    private $locator;

    /** @var SettingsInterface */
    private $settings;

    /** @var CanonicalUrl */
    private $canonical;

    /** @var string */
    private $extension = 'tpl';

    public function __construct(DWebSmarty $smarty, TemplateLocatorInterface $locator, SettingsInterface $settings, CanonicalUrl $canonical)
    {
        $this->smarty   = $smarty;
        $this->locator  = $locator;
        $this->settings = $settings;
        $this->canonical = $canonical;
    }

    public function render($moduleName, $templateId, array $data = array())
    {
        $path = $this->locator->resolve($moduleName, $templateId, $this->extension);
        if (!$path) {
            throw new HttpException(
                404,
                'Not Found',
                "Template not found: {$moduleName}::{$templateId}.{$this->extension}"
            );
        }

        // Clear previous assigns to avoid leakage across renders.
        $this->smarty->clearAllAssign();

        $sections = new \boru\dweb\Rendering\SectionBag();
        $assets   = new \boru\dweb\Assets\AssetManager($this->settings, $sections);

        // Provide helper as $dweb (explicit, override-aware)
        $basePath = '';
        if (isset($_SERVER['SCRIPT_NAME'])) {
            $dir = rtrim(str_replace('\\', '/', dirname((string)$_SERVER['SCRIPT_NAME'])), '/');
            if ($dir === '.' || $dir === '\\' || $dir === '/') $dir = '';
            $basePath = $dir;
        }
        $this->smarty->assign('dweb', new TemplateHelper(
            $this->locator,
            $moduleName,
            $basePath,
            $this->canonical,
            $sections,
            $assets
        ));

        foreach ($data as $k => $v) {
            $this->smarty->assign((string)$k, $v);
        }

        // Render by absolute path
        return $this->smarty->fetch('file:' . $path);
    }

    public function renderLayout($moduleName, $layoutId, $templateId, array $data = array())
    {
        $content = $this->render($moduleName, $templateId, $data);

        $layoutData = $data;
        $layoutData['content'] = $content;

        return $this->render($moduleName, $layoutId, $layoutData);
    }

    /**
     * Helper: build a configured Smarty instance based on dweb settings.
     */
    public static function buildSmarty(SettingsInterface $settings)
    {
        $compileDir = (string)$settings->get('dweb.smarty.compile_dir', sys_get_temp_dir() . '/dweb_smarty/compile');
        $cacheDir   = (string)$settings->get('dweb.smarty.cache_dir',   sys_get_temp_dir() . '/dweb_smarty/cache');

        self::ensureDir($compileDir);
        self::ensureDir($cacheDir);

        $smarty = new DWebSmarty();
        $smarty->setCompileDir($compileDir);
        $smarty->setCacheDir($cacheDir);

        // sane defaults
        $smarty->setCompileCheck((bool)$settings->get('dweb.smarty.compile_check', true));
        $smarty->setCaching((bool)$settings->get('dweb.smarty.caching', false));
        $smarty->setCacheLifetime((int)$settings->get('dweb.smarty.cache_lifetime', 0));

        // Security note: no template_dir needed since we render via file: absolute path
        return $smarty;
    }

    private static function ensureDir($dir)
    {
        $dir = rtrim((string)$dir, DIRECTORY_SEPARATOR);
        if ($dir === '') return;

        if (is_dir($dir)) return;

        if (!@mkdir($dir, 0775, true) && !is_dir($dir)) {
            throw new HttpException(
                503,
                'Service Unavailable',
                'Smarty directories could not be initialized.',
                "Failed to create directory: {$dir}"
            );
        }
    }
}
