<?php
namespace boru\dweb\Rendering;

use boru\dweb\Contracts\TemplateLocatorInterface;
use boru\dweb\Routing\CanonicalUrl;

/**
 * Template helper exposed to templates as $dweb.
 *
 * Purpose:
 * - Resolve override-aware template paths explicitly (no magic inference).
 * - Provide Smarty-friendly include strings ("file:/abs/path.tpl").
 * - Provide PHP-friendly absolute paths for require/include.
 */
class TemplateHelper
{
    /** @var TemplateLocatorInterface */
    private $locator;

    /** @var string */
    private $moduleName;

    /** @var string */
    private $smartyExt = 'tpl';

    /** @var string */
    private $phpExt = 'php';

    /** @var string */
    private $basePath;

    /** @var CanonicalUrl */
    private $canonical;

    private $sections;
    private $assets;

    public function __construct(TemplateLocatorInterface $locator, $moduleName, $basePath, CanonicalUrl $canonical, $sections, $assets)
    {
        $this->locator    = $locator;
        $this->moduleName = (string)$moduleName;
        $this->basePath   = (string)$basePath;
        $this->canonical  = $canonical;
        $this->sections   = $sections;
        $this->assets     = $assets;
    }

    public function basePath()
    {
        return $this->basePath;
    }

    public function url($path)
    {
        $path = (string)$path;
        if ($path === '') return $this->basePath !== '' ? $this->basePath . '/' : '/';
        if ($path[0] !== '/') $path = '/' . $path;
        return $this->basePath !== '' ? ($this->basePath . $path) : $path;
    }

    /**
     * Resolve a template id to an absolute path (override-aware).
     *
     * @param string $templateId e.g. "partials/hello"
     * @param string|null $moduleName optional different module
     * @param string $extension "tpl" or "php"
     * @return string|null
     */
    public function path($templateId, $moduleName = null, $extension = 'tpl')
    {
        $module = $moduleName !== null ? (string)$moduleName : $this->moduleName;
        return $this->locator->resolve($module, (string)$templateId, (string)$extension);
    }

    /**
     * Smarty include string ("file:/abs/path.tpl") or null if missing.
     *
     * Usage in Smarty:
     *   {include file=$dweb->file('partials/hello')}
     *
     * @param string $templateId
     * @param string|null $moduleName
     * @return string|null
     */
    public function file($templateId, $moduleName = null)
    {
        $path = $this->path($templateId, $moduleName, $this->smartyExt);
        return $path ? ('file:' . $path) : null;
    }

    /**
     * Convenience: resolve "partials/<id>"
     *
     * @param string $partialId e.g. "hello"
     * @param string|null $moduleName
     * @return string|null
     */
    public function partialFile($partialId, $moduleName = null)
    {
        return $this->file('partials/' . (string)$partialId, $moduleName);
    }

    /**
     * Convenience: resolve "fragments/<id>"
     *
     * @param string $fragmentId e.g. "hello"
     * @param string|null $moduleName
     * @return string|null
     */
    public function fragmentFile($fragmentId, $moduleName = null)
    {
        return $this->file('fragments/' . (string)$fragmentId, $moduleName);
    }

    /**
     * PHP-template convenience for require/include.
     *
     * Usage in PHP templates:
     *   $p = $dweb->phpPath('partials/hello');
     *   if ($p) require $p;
     *
     * @param string $templateId
     * @param string|null $moduleName
     * @return string|null
     */
    public function phpPath($templateId, $moduleName = null)
    {
        return $this->path($templateId, $moduleName, $this->phpExt);
    }

    public function viewUrl($qualified)
    {
        return $this->canonical->view($qualified);
    }

    public function actionUrl($qualified)
    {
        return $this->canonical->action($qualified);
    }

    /**
     * Module-local sugar: view URL within the current module.
     *
     * Smarty:
     *   {$dweb->view('home')}
     *
     * @param string $viewName e.g. "home", "another", "helloFragment"
     * @return string
     */
    public function view($viewName)
    {
        $qualified = $this->canonical->qualifyView($this->moduleName, (string)$viewName);
        return $this->canonical->view($qualified);
    }

    /**
     * Module-local sugar: action URL within the current module.
     *
     * Smarty:
     *   {$dweb->action('echo')}
     *
     * @param string $actionName e.g. "ping", "echo"
     * @return string
     */
    public function action($actionName)
    {
        $qualified = $this->canonical->qualifyAction($this->moduleName, (string)$actionName);
        return $this->canonical->action($qualified);
    }

    /**
     * Optional cross-module sugar (still explicit):
     *   {$dweb->mView('Other', 'home')}
     *   {$dweb->mAction('Other', 'echo')}
     */
    public function mView($moduleName, $viewName)
    {
        $qualified = $this->canonical->qualifyView((string)$moduleName, (string)$viewName);
        return $this->canonical->view($qualified);
    }

    public function mAction($moduleName, $actionName)
    {
        $qualified = $this->canonical->qualifyAction((string)$moduleName, (string)$actionName);
        return $this->canonical->action($qualified);
    }

    public function canonicalEnabled()
    {
        return $this->canonical->enabled();
    }

    public function module()
    {
        return $this->moduleName;
    }

    public function assets()
    {
        return $this->assets;
    }

    public function section($name, $default = '')
    {
        if ($this->sections === null) return (string)$default;

        $html = $this->sections->get((string)$name, (string)$default);
        return $html === '' ? '' : ($html . "\n");
    }

    /**
     * Convention selector for a section name: "#section-<name>".
     * Useful in templates for hx-target, anchors, etc.
     *
     * @param string $sectionName
     * @return string
     */
    public function sectionSelector($sectionName)
    {
        return '#section-' . (string)$sectionName;
    }

    /**
     * Convenience: stable DOM id for a section ("section-<name>").
     *
     * @param string $sectionName
     * @return string
     */
    public function sectionId($sectionName)
    {
        return 'section-' . (string)$sectionName;
    }
}
